<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/jwt_helper.php';
require_once __DIR__ . '/vendor/autoload.php';

$endpoint = isset($_GET['endpoint']) ? $_GET['endpoint'] : '';

// Function to get input data
function getInputData()
{
    $input = file_get_contents('php://input');
    return json_decode($input, true) ?? [];
}

// REGISTER
if ($endpoint == 'register') {
    $input = getInputData();

    $nama = isset($input['nama']) ? trim($input['nama']) : '';
    $email = isset($input['email']) ? trim($input['email']) : '';
    $password = isset($input['password']) ? $input['password'] : '';
    $ulangi_password = isset($input['ulangi_password']) ? $input['ulangi_password'] : '';

    // Validasi input
    if (empty($nama) || empty($email) || empty($password) || empty($ulangi_password)) {
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Semua field wajib diisi",
            "error_code" => "MISSING_FIELDS"
        ]);
        exit;
    }

    if ($password !== $ulangi_password) {
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Konfirmasi password tidak sama",
            "error_code" => "PASSWORD_MISMATCH"
        ]);
        exit;
    }

    // Cek email sudah terdaftar
    $cek = $conn->prepare("SELECT id FROM users WHERE email = ?");
    $cek->bind_param("s", $email);
    $cek->execute();
    $cek->store_result();
    if ($cek->num_rows > 0) {
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Email sudah terdaftar",
            "error_code" => "EMAIL_EXISTS"
        ]);
        exit;
    }

    // Simpan user baru ke MySql
    $password_hash = password_hash($password, PASSWORD_DEFAULT);
    $tanggal_dibuat = date('Y-m-d H:i:s');
    $sql = "INSERT INTO users (nama, email, password_hash, tanggal_dibuat) VALUES (?, ?, ?, ?)";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ssss", $nama, $email, $password_hash, $tanggal_dibuat);
    $result = $stmt->execute();

    if ($result) {
        $id = $conn->insert_id;

        // Generate JWT token untuk user baru
        $user_data = [
            'id' => $id,
            'nama' => $nama,
            'email' => $email
        ];

        $token = JWTHelper::generateToken($user_data);

        http_response_code(201);
        echo json_encode([
            "status" => "success",
            "message" => "Registrasi berhasil",
            "data" => [
                "user" => [
                    "id" => $id,
                    "nama" => $nama,
                    "email" => $email,
                    "tanggal_dibuat" => $tanggal_dibuat
                ],
                "token" => $token,
                "token_type" => "Bearer"
            ]
        ]);
    } else {
        http_response_code(500);
        echo json_encode([
            "status" => "error",
            "message" => "Registrasi gagal",
            "error_code" => "REGISTRATION_FAILED"
        ]);
    }
    exit;
}

// LOGIN
if ($endpoint == 'login') {
    $input = getInputData();

    $email = isset($input['email']) ? $input['email'] : '';
    $password = isset($input['password']) ? $input['password'] : '';

    if (empty($email) || empty($password)) {
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Email dan password wajib diisi",
            "error_code" => "MISSING_CREDENTIALS"
        ]);
        exit;
    }

    // Query user
    $sql = "SELECT * FROM users WHERE email = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $email);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($row = $result->fetch_assoc()) {
        // Verifikasi password hash
        if (password_verify($password, $row['password_hash'])) {
            // Generate JWT token
            $user_data = [
                'id' => $row['id'],
                'nama' => $row['nama'],
                'email' => $row['email']
            ];

            $token = JWTHelper::generateToken($user_data);

            echo json_encode([
                "status" => "success",
                "message" => "Login berhasil",
                "data" => [
                    "user" => [
                        "id" => $row['id'],
                        "nama" => $row['nama'],
                        "email" => $row['email'],
                        "tanggal_dibuat" => $row['tanggal_dibuat']
                    ],
                    "token" => $token,
                    "token_type" => "Bearer"
                ]
            ]);
        } else {
            http_response_code(401);
            echo json_encode([
                "status" => "error",
                "message" => "Password salah",
                "error_code" => "INVALID_PASSWORD"
            ]);
        }
    } else {
        http_response_code(401);
        echo json_encode([
            "status" => "error",
            "message" => "Email tidak ditemukan",
            "error_code" => "EMAIL_NOT_FOUND"
        ]);
    }
    exit;
}

// REFRESH TOKEN
if ($endpoint == 'refresh') {
    $user_payload = JWTHelper::requireAuth();

    // Generate new token
    $user_data = [
        'id' => $user_payload['user_id'],
        'nama' => $user_payload['nama'],
        'email' => $user_payload['email']
    ];

    $new_token = JWTHelper::generateToken($user_data);

    echo json_encode([
        "status" => "success",
        "message" => "Token berhasil di-refresh",
        "data" => [
            "token" => $new_token,
            "token_type" => "Bearer"
        ]
    ]);
    exit;
}

// PROFILE - Get user profile
if ($endpoint == 'profile') {
    $user_payload = JWTHelper::requireAuth();

    // Get fresh user data from database
    $sql = "SELECT id, nama, email, tanggal_dibuat FROM users WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $user_payload['user_id']);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($row = $result->fetch_assoc()) {
        echo json_encode([
            "status" => "success",
            "message" => "Profile berhasil diambil",
            "data" => $row
        ]);
    } else {
        http_response_code(404);
        echo json_encode([
            "status" => "error",
            "message" => "User tidak ditemukan",
            "error_code" => "USER_NOT_FOUND"
        ]);
    }
    exit;
}

// UPDATE PROFILE - Edit nama user (PUT method)
if ($endpoint == 'update-profile' && $_SERVER['REQUEST_METHOD'] === 'PUT') {
    $user_payload = JWTHelper::requireAuth();
    $input = getInputData();

    $nama = isset($input['nama']) ? trim($input['nama']) : '';

    // Validasi input
    if (empty($nama)) {
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Nama tidak boleh kosong",
            "error_code" => "MISSING_NAME"
        ]);
        exit;
    }

    // Validasi panjang nama
    if (strlen($nama) < 3) {
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Nama minimal 3 karakter",
            "error_code" => "NAME_TOO_SHORT"
        ]);
        exit;
    }

    if (strlen($nama) > 100) {
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Nama maksimal 100 karakter",
            "error_code" => "NAME_TOO_LONG"
        ]);
        exit;
    }

    // Update nama di database
    $sql = "UPDATE users SET nama = ? WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("si", $nama, $user_payload['user_id']);
    $result = $stmt->execute();

    if ($result) {
        // Get updated user data
        $sql = "SELECT id, nama, email, tanggal_dibuat FROM users WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $user_payload['user_id']);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();

        // Generate new token with updated name
        $user_data = [
            'id' => $row['id'],
            'nama' => $row['nama'],
            'email' => $row['email']
        ];

        $new_token = JWTHelper::generateToken($user_data);

        echo json_encode([
            "status" => "success",
            "message" => "Nama berhasil diperbarui",
            "data" => [
                "user" => $row,
                "token" => $new_token,
                "token_type" => "Bearer"
            ]
        ]);
    } else {
        http_response_code(500);
        echo json_encode([
            "status" => "error",
            "message" => "Gagal memperbarui nama",
            "error_code" => "UPDATE_FAILED"
        ]);
    }
    exit;
}

// LOGOUT (Optional - for client side token removal)
if ($endpoint == 'logout') {
    echo json_encode([
        "status" => "success",
        "message" => "Logout berhasil"
    ]);
    exit;
}

// Endpoint not found
http_response_code(404);
echo json_encode([
    "status" => "error",
    "message" => "Endpoint tidak ditemukan",
    "error_code" => "ENDPOINT_NOT_FOUND"
]);
