<?php
// filepath: d:\laragon\www\tani-api\jwt_helper.php
require_once __DIR__ . '/config.php';

class JWTHelper
{
    private static function getSecretKey()
    {
        return Config::get('JWT_SECRET', 'fallback-secret-key-for-development');
    }
    
    private static function getExpiry()
    {
        return (int)Config::get('JWT_EXPIRY', 86400); // Default 24 hours
    }
    
    private static $algorithm = 'HS256';

    // Generate JWT Token
    public static function generateToken($user_data)
    {
        $current_time = time();
        $expiry_time = $current_time + self::getExpiry();

        error_log("🔐 [JWT] Generating token for user: " . $user_data['id']);

        $header = json_encode(['typ' => 'JWT', 'alg' => self::$algorithm]);

        $payload = json_encode([
            'iss' => 'tani-api',
            'aud' => 'tani-app',
            'iat' => $current_time,
            'exp' => $expiry_time,
            'user_id' => $user_data['id'],
            'email' => $user_data['email'],
            'nama' => $user_data['nama']
        ]);

        $base64Header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
        $base64Payload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));

        $signature = hash_hmac('sha256', $base64Header . "." . $base64Payload, self::getSecretKey(), true);
        $base64Signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));

        return $base64Header . "." . $base64Payload . "." . $base64Signature;
    }

    public static function verifyToken($token)
    {
        if (!$token) return false;

        $tokenParts = explode('.', $token);
        if (count($tokenParts) != 3) return false;

        $header = base64_decode(str_replace(['-', '_'], ['+', '/'], $tokenParts[0]));
        $payload = base64_decode(str_replace(['-', '_'], ['+', '/'], $tokenParts[1]));
        $signatureProvided = $tokenParts[2];

        $base64Header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
        $base64Payload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
        $signature = hash_hmac('sha256', $base64Header . "." . $base64Payload, self::getSecretKey(), true);
        $base64Signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));

        if (!hash_equals($base64Signature, $signatureProvided)) return false;

        $payloadData = json_decode($payload, true);
        if (!$payloadData || $payloadData['exp'] < time()) return false;

        return $payloadData;
    }

    // ... sisa method tetap sama
    public static function getTokenFromHeader()
    {
        if (function_exists('getallheaders')) {
            $headers = getallheaders();
            foreach ($headers as $key => $value) {
                if (strtolower($key) === 'authorization') {
                    if (preg_match('/Bearer\s+(\S+)/', $value, $matches)) {
                        return $matches[1];
                    }
                }
            }
        }

        $authHeader = $_SERVER['HTTP_AUTHORIZATION'] ?? $_SERVER['REDIRECT_HTTP_AUTHORIZATION'] ?? null;
        if ($authHeader && preg_match('/Bearer\s+(\S+)/', $authHeader, $matches)) {
            return $matches[1];
        }

        return null;
    }

    public static function requireAuth()
    {
        $token = self::getTokenFromHeader();
        $payload = self::verifyToken($token);

        if (!$payload) {
            http_response_code(401);
            echo json_encode([
                "status" => "error",
                "message" => "Token tidak valid atau sudah expired",
                "error_code" => "UNAUTHORIZED"
            ]);
            exit;
        }

        return $payload;
    }
}