<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

// Enable error logging
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/api_debug.log');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/jwt_helper.php';

// Get HTTP method
$method = $_SERVER['REQUEST_METHOD'];

// Function to get input data
function getInputData()
{
    $input = file_get_contents('php://input');
    error_log("📥 [KEBUN] Raw input: " . $input);

    $data = json_decode($input, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("❌ [KEBUN] JSON decode error: " . json_last_error_msg());
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Invalid JSON format: " . json_last_error_msg(),
            "error_code" => "INVALID_JSON",
            "timestamp" => date('Y-m-d H:i:s')
        ]);
        exit;
    }

    error_log("✅ [KEBUN] Parsed data: " . json_encode($data));
    return $data ?? [];
}

// Function untuk response yang konsisten
function sendResponse($status, $message, $data = null, $httpCode = 200, $errorCode = null)
{
    error_log("📤 [KEBUN] Sending response: Status=$status, Code=$httpCode, Message=$message");

    http_response_code($httpCode);
    $response = [
        "status" => $status,
        "message" => $message,
        "timestamp" => date('Y-m-d H:i:s')
    ];

    if ($data !== null) {
        $response["data"] = $data;
    }

    if ($errorCode !== null) {
        $response["error_code"] = $errorCode;
    }

    // echo json_encode($response);
    echo json_encode($response, JSON_PRESERVE_ZERO_FRACTION);
    exit;
}

// Get ID from URL if exists
$path = $_SERVER['REQUEST_URI'];
$path_parts = explode('/', trim($path, '/'));
$id = null;

// Extract ID from URL path
foreach ($path_parts as $part) {
    if (is_numeric($part)) {
        $id = (int)$part;
        break;
    }
}

error_log("🔍 [KEBUN] Method: $method, Path: $path, ID: " . ($id ?? 'null'));

// ✅ AUTH CHECK - Get current user
try {
    $user_payload = JWTHelper::requireAuth();
    $current_user_id = $user_payload['user_id'];
    error_log("👤 [KEBUN] Authenticated user ID: $current_user_id");
} catch (Exception $e) {
    error_log("🚨 [KEBUN] Auth failed: " . $e->getMessage());
    sendResponse("error", "Token tidak valid atau telah expire", null, 401, "UNAUTHORIZED");
}

// GET - Read operations
if ($method == 'GET') {
    try {
        if ($id) {
            // GET /kebun/1 - Read by ID (hanya milik user yang login)
            error_log("🔍 [KEBUN] Getting kebun by ID: $id for user: $current_user_id");

            $sql = "SELECT l.*, u.nama as nama_user FROM data_kebun l JOIN users u ON l.user_id = u.id WHERE l.id = ? AND l.user_id = ?";
            $stmt = $conn->prepare($sql);

            if (!$stmt) {
                error_log("❌ [KEBUN] Prepare failed: " . $conn->error);
                sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
            }

            $stmt->bind_param("ii", $id, $current_user_id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($row = $result->fetch_assoc()) {
                error_log("✅ [KEBUN] Found kebun: " . $row['nama']);
                sendResponse("success", "Data kebun berhasil ditemukan", $row);
            } else {
                error_log("❌ [KEBUN] Kebun not found or not owned by user");
                sendResponse("error", "Kebun tidak ditemukan atau bukan milik Anda", null, 404, "KEBUN_NOT_FOUND");
            }
        } else {
            // GET /kebun - Read semua kebun milik user yang login
            error_log("📋 [KEBUN] Getting all kebun for user: $current_user_id");
            $sql = "SELECT l.*, u.nama as nama_user
                    FROM data_kebun l 
                    JOIN users u ON l.user_id = u.id
                    WHERE l.user_id = ? 
                    ORDER BY l.id DESC";
            $stmt = $conn->prepare($sql);

            if (!$stmt) {
                error_log("❌ [KEBUN] Prepare failed: " . $conn->error);
                sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
            }

            $stmt->bind_param("i", $current_user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $data = [];

            // ✅ Hitung total luas dan total titik tanam
            $total_luas = 0;
            $total_titik_tanam = 0;

            while ($row = $result->fetch_assoc()) {
                $total_luas += (float)$row['luas'];
                $total_titik_tanam += (int)$row['titik_tanam'];
                $data[] = $row;
            }

            error_log("✅ [KEBUN] Found " . count($data) . " kebun records");

            $metadata = [
                "total_records" => count($data),
                // "total_luas" => $total_luas,
                "total_luas" => round((float)$total_luas, 1),
                "total_titik_tanam" => $total_titik_tanam,
                "user_id" => $current_user_id,
                "user_name" => $user_payload['nama']
            ];

            error_log("📊 [KEBUN] Total Luas: $total_luas, Total Titik Tanam: $total_titik_tanam");

            sendResponse("success", "Data kebun berhasil diambil", [
                "kebun" => $data,
                "metadata" => $metadata
            ]);
        }
    } catch (Exception $e) {
        error_log("💥 [KEBUN] GET Exception: " . $e->getMessage());
        sendResponse("error", "Terjadi kesalahan sistem", null, 500, "SYSTEM_ERROR");
    }
}

// POST - Create new kebun (otomatis assign ke user yang login)
if ($method == 'POST') {
    try {
        error_log("🌱 [KEBUN] Creating new kebun for user: $current_user_id");

        $input = getInputData();

        // Validation
        $required_fields = ['nama', 'lokasi', 'luas', 'titik_tanam', 'waktu_beli', 'status_kepemilikan', 'status_kebun'];
        $missing_fields = [];

        foreach ($required_fields as $field) {
            if (!isset($input[$field]) || $input[$field] === '' || $input[$field] === null) {
                $missing_fields[] = $field;
            }
        }

        if (!empty($missing_fields)) {
            error_log("❌ [KEBUN] Missing fields: " . implode(', ', $missing_fields));
            sendResponse("error", "Field berikut wajib diisi: " . implode(', ', $missing_fields), null, 400, "MISSING_REQUIRED_FIELDS");
        }

        // Sanitize and validate data
        $nama = trim($input['nama']);
        $lokasi = trim($input['lokasi']);
        $luas = trim($input['luas']);
        $titik_tanam = (int)$input['titik_tanam'];
        $waktu_beli_raw = trim($input['waktu_beli']);
        $status_kepemilikan = trim($input['status_kepemilikan']);
        $status_kebun = trim($input['status_kebun']);

        // ✅ FIX: Convert date format from MM/dd/yyyy to yyyy-MM-dd
        $waktu_beli = convertDateFormat($waktu_beli_raw);
        if (!$waktu_beli) {
            error_log("❌ [KEBUN] Invalid date format: " . $waktu_beli_raw);
            sendResponse("error", "Format tanggal tidak valid. Gunakan format DD/MM/YYYY atau YYYY-MM-DD", null, 400, "INVALID_DATE_FORMAT");
        }

        error_log("✅ [KEBUN] Date converted: $waktu_beli_raw -> $waktu_beli");
        // Additional validation
        if (strlen($nama) < 2) {
            sendResponse("error", "Nama kebun minimal 2 karakter", null, 400, "INVALID_NAME");
        }

        if (strlen($lokasi) < 3) {
            sendResponse("error", "Lokasi minimal 3 karakter", null, 400, "INVALID_LOCATION");
        }

        if ($titik_tanam <= 0) {
            sendResponse("error", "Titik tanam harus lebih dari 0", null, 400, "INVALID_TITIK_TANAM");
        }

        error_log("✅ [KEBUN] Data validated successfully");
        error_log("📊 [KEBUN] Data to insert: nama=$nama, lokasi=$lokasi, luas=$luas, titik_tanam=$titik_tanam, waktu_beli=$waktu_beli");

        // Insert data kebun dengan user_id dari token
        $sql = "INSERT INTO data_kebun (user_id, nama, lokasi, luas, titik_tanam, waktu_beli, status_kepemilikan, status_kebun) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            error_log("❌ [KEBUN] Prepare failed: " . $conn->error);
            sendResponse("error", "Database prepare error: " . $conn->error, null, 500, "DATABASE_PREPARE_ERROR");
        }

        // Use correct bind_param types
        $bind_result = $stmt->bind_param("isssssss", $current_user_id, $nama, $lokasi, $luas, $titik_tanam, $waktu_beli, $status_kepemilikan, $status_kebun);

        if (!$bind_result) {
            error_log("❌ [KEBUN] Bind param failed: " . $stmt->error);
            sendResponse("error", "Database bind error: " . $stmt->error, null, 500, "DATABASE_BIND_ERROR");
        }

        $result = $stmt->execute();

        if (!$result) {
            error_log("❌ [KEBUN] Execute failed: " . $stmt->error);
            sendResponse("error", "Database execute error: " . $stmt->error, null, 500, "DATABASE_EXECUTE_ERROR");
        }

        if ($result) {
            $new_id = $conn->insert_id;
            error_log("✅ [KEBUN] Successfully created kebun with ID: $new_id");

            $response_data = [
                "id" => $new_id,
                "user_id" => $current_user_id,
                "nama" => $nama,
                "lokasi" => $lokasi,
                "luas" => $luas,
                "titik_tanam" => $titik_tanam,
                "waktu_beli" => $waktu_beli,
                "status_kepemilikan" => $status_kepemilikan,
                "status_kebun" => $status_kebun,
                "nama_user" => $user_payload['nama']
            ];

            sendResponse("success", "Kebun berhasil ditambahkan", $response_data, 201);
        } else {
            error_log("❌ [KEBUN] Insert failed but no error message");
            sendResponse("error", "Gagal menambahkan kebun", null, 500, "DATABASE_ERROR");
        }
    } catch (Exception $e) {
        error_log("💥 [KEBUN] POST Exception: " . $e->getMessage());
        error_log("💥 [KEBUN] Stack trace: " . $e->getTraceAsString());
        sendResponse("error", "Terjadi kesalahan sistem: " . $e->getMessage(), null, 500, "SYSTEM_ERROR");
    }
}

// ✅ ADD: Function to convert date format
function convertDateFormat($dateString)
{
    error_log("🔄 [DATE] Converting date: " . $dateString);

    // Try different date formats
    $formats = [
        'Y-m-d',        // 2025-10-03 (already correct)
        'd/m/Y',        // 03/10/2025 (DD/MM/YYYY)
        'm/d/Y',        // 10/03/2025 (MM/DD/YYYY)
        'd-m-Y',        // 03-10-2025 (DD-MM-YYYY)
        'm-d-Y',        // 10-03-2025 (MM-DD-YYYY)
    ];

    foreach ($formats as $format) {
        $date = DateTime::createFromFormat($format, $dateString);
        if ($date && $date->format($format) === $dateString) {
            $converted = $date->format('Y-m-d');
            error_log("✅ [DATE] Successfully converted using format $format: $dateString -> $converted");
            return $converted;
        }
    }

    error_log("❌ [DATE] Failed to convert date: " . $dateString);
    return false;
}

// // PUT - Update kebun (hanya milik user yang login)
if ($method == 'PUT') {
    try {
        if (!$id) {
            sendResponse("error", "ID kebun diperlukan untuk update", null, 400, "MISSING_ID");
        }

        error_log("🔄 [KEBUN] Updating kebun ID: $id for user: $current_user_id");

        $input = getInputData();

        // Validation
        $required_fields = ['nama', 'lokasi', 'luas', 'titik_tanam', 'waktu_beli', 'status_kepemilikan', 'status_kebun'];
        $missing_fields = [];

        foreach ($required_fields as $field) {
            if (!isset($input[$field]) || $input[$field] === '' || $input[$field] === null) {
                $missing_fields[] = $field;
            }
        }

        if (!empty($missing_fields)) {
            error_log("❌ [KEBUN] Missing fields: " . implode(', ', $missing_fields));
            sendResponse("error", "Field berikut wajib diisi: " . implode(', ', $missing_fields), null, 400, "MISSING_REQUIRED_FIELDS");
        }

        // Cek apakah kebun exists dan milik user yang login
        $cek = $conn->prepare("SELECT id FROM data_kebun WHERE id = ? AND user_id = ?");
        $cek->bind_param("ii", $id, $current_user_id);
        $cek->execute();
        $cek->store_result();

        if ($cek->num_rows == 0) {
            error_log("❌ [KEBUN] Kebun not found or not owned by user");
            sendResponse("error", "Kebun tidak ditemukan atau bukan milik Anda", null, 404, "KEBUN_NOT_FOUND");
        }

        // Sanitize and validate data
        $nama = trim($input['nama']);
        $lokasi = trim($input['lokasi']);
        $luas = trim($input['luas']);
        $titik_tanam = (int)$input['titik_tanam'];
        $waktu_beli_raw = trim($input['waktu_beli']);
        $status_kepemilikan = trim($input['status_kepemilikan']);
        $status_kebun = trim($input['status_kebun']);

        // ✅ Convert date format
        $waktu_beli = convertDateFormat($waktu_beli_raw);
        if (!$waktu_beli) {
            error_log("❌ [KEBUN] Invalid date format: " . $waktu_beli_raw);
            sendResponse("error", "Format tanggal tidak valid. Gunakan format DD-MM-YYYY atau YYYY-MM-DD", null, 400, "INVALID_DATE_FORMAT");
        }

        error_log("✅ [KEBUN] Date converted: $waktu_beli_raw -> $waktu_beli");
        // Additional validation
        if (strlen($nama) < 2) {
            sendResponse("error", "Nama kebun minimal 2 karakter", null, 400, "INVALID_NAME");
        }

        if (strlen($lokasi) < 3) {
            sendResponse("error", "Lokasi minimal 3 karakter", null, 400, "INVALID_LOCATION");
        }

        if ($titik_tanam <= 0) {
            sendResponse("error", "Titik tanam harus lebih dari 0", null, 400, "INVALID_TITIK_TANAM");
        }

        error_log("✅ [KEBUN] Data validated successfully for update");
        error_log("📊 [KEBUN] Data to update:");
        error_log("   - nama: $nama");
        error_log("   - lokasi: $lokasi");
        error_log("   - luas: $luas");
        error_log("   - titik_tanam: $titik_tanam");
        error_log("   - waktu_beli: $waktu_beli");
        error_log("   - status_kepemilikan: $status_kepemilikan");
        error_log("   - status_kebun: $status_kebun");
        error_log("   - id: $id");
        error_log("   - user_id: $current_user_id");

        // ✅ FIX: Update data kebun dengan bind_param yang BENAR
        $sql = "UPDATE data_kebun 
                SET nama = ?, 
                    lokasi = ?, 
                    luas = ?, 
                    titik_tanam = ?, 
                    waktu_beli = ?, 
                    status_kepemilikan = ?, 
                    status_kebun = ? 
                WHERE id = ? AND user_id = ?";

        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            error_log("❌ [KEBUN] Update prepare failed: " . $conn->error);
            sendResponse("error", "Database error: " . $conn->error, null, 500, "DATABASE_ERROR");
        }

        // ✅ FIX: Correct bind_param types dan urutan
        // SQL order: nama, lokasi, luas, titik_tanam, waktu_beli, status_kepemilikan, status_kebun, id, user_id
        // Types:     s     s       s     i            s           s                   s             i   i
        $stmt->bind_param(
            "sssisssii",
            $nama,                // s - string
            $lokasi,              // s - string
            $luas,                // s - string
            $titik_tanam,         // i - integer ✅ FIX
            $waktu_beli,          // s - string
            $status_kepemilikan,  // s - string
            $status_kebun,        // s - string
            $id,                  // i - integer
            $current_user_id      // i - integer
        );

        $result = $stmt->execute();

        if ($result) {
            if ($stmt->affected_rows > 0) {
                error_log("✅ [KEBUN] Successfully updated kebun ID: $id");

                $response_data = [
                    "id" => $id,
                    "user_id" => $current_user_id,
                    "nama" => $nama,
                    "lokasi" => $lokasi,
                    "luas" => $luas,
                    "titik_tanam" => $titik_tanam,
                    "waktu_beli" => $waktu_beli,
                    "status_kepemilikan" => $status_kepemilikan,
                    "status_kebun" => $status_kebun,
                    "nama_user" => $user_payload['nama']
                ];

                sendResponse("success", "Kebun berhasil diupdate", $response_data);
            } else {
                error_log("⚠️ [KEBUN] No rows affected - data might be the same");
                sendResponse("error", "Tidak ada perubahan data", null, 400, "NO_CHANGES");
            }
        } else {
            error_log("❌ [KEBUN] Update failed: " . $stmt->error);
            sendResponse("error", "Gagal mengupdate kebun: " . $stmt->error, null, 500, "UPDATE_FAILED");
        }
    } catch (Exception $e) {
        error_log("💥 [KEBUN] PUT Exception: " . $e->getMessage());
        error_log("💥 [KEBUN] Stack trace: " . $e->getTraceAsString());
        sendResponse("error", "Terjadi kesalahan sistem: " . $e->getMessage(), null, 500, "SYSTEM_ERROR");
    }
}

// DELETE - Delete kebun (hanya milik user yang login)
if ($method == 'DELETE') {
    try {
        if (!$id) {
            sendResponse("error", "ID kebun diperlukan untuk delete", null, 400, "MISSING_ID");
        }

        error_log("🗑️ [KEBUN] Deleting kebun ID: $id for user: $current_user_id");

        // Cek apakah kebun exists dan milik user yang login
        $cek = $conn->prepare("SELECT id FROM data_kebun WHERE id = ? AND user_id = ?");
        $cek->bind_param("ii", $id, $current_user_id);
        $cek->execute();
        $cek->store_result();

        if ($cek->num_rows == 0) {
            sendResponse("error", "Kebun tidak ditemukan atau bukan milik Anda", null, 404, "KEBUN_NOT_FOUND");
        }

        // Hapus kebun (hanya milik user yang login)
        $sql = "DELETE FROM data_kebun WHERE id = ? AND user_id = ?";
        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            error_log("❌ [KEBUN] Delete prepare failed: " . $conn->error);
            sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
        }

        $stmt->bind_param("ii", $id, $current_user_id);
        $result = $stmt->execute();

        if ($result && $stmt->affected_rows > 0) {
            error_log("✅ [KEBUN] Successfully deleted kebun ID: $id");
            sendResponse("success", "Kebun berhasil dihapus");
        } else {
            error_log("❌ [KEBUN] Delete failed or no rows affected");
            sendResponse("error", "Kebun tidak ditemukan atau bukan milik Anda", null, 404, "KEBUN_NOT_FOUND");
        }
    } catch (Exception $e) {
        error_log("💥 [KEBUN] DELETE Exception: " . $e->getMessage());
        sendResponse("error", "Terjadi kesalahan sistem", null, 500, "SYSTEM_ERROR");
    }
}

// Method not allowed
error_log("❌ [KEBUN] Method not allowed: $method");
sendResponse("error", "Method tidak didukung", null, 405, "METHOD_NOT_ALLOWED");
