<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/api_debug.log');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/jwt_helper.php';

function sendResponse($status, $message, $data = null, $httpCode = 200, $errorCode = null)
{
    http_response_code($httpCode);
    $response = [
        "status" => $status,
        "message" => $message,
        "timestamp" => date('Y-m-d H:i:s')
    ];
    if ($data !== null) $response["data"] = $data;
    if ($errorCode !== null) $response["error_code"] = $errorCode;
    echo json_encode($response);
    exit;
}

// AUTH CHECK
try {
    $user_payload = JWTHelper::requireAuth();
    $current_user_id = $user_payload['user_id'];
    error_log("👤 [LAPORAN-DATA] Authenticated user ID: $current_user_id");
} catch (Exception $e) {
    sendResponse("error", "Token tidak valid atau telah expire", null, 401, "UNAUTHORIZED");
}

// GET - Retrieve Laporan Data
if ($_SERVER['REQUEST_METHOD'] == 'GET') {
    try {
        // Get query parameters
        $kebun_id = isset($_GET['kebun_id']) ? (int)$_GET['kebun_id'] : null;
        $tanggal_dari = isset($_GET['tanggal_dari']) ? $_GET['tanggal_dari'] : null;
        $tanggal_sampai = isset($_GET['tanggal_sampai']) ? $_GET['tanggal_sampai'] : null;

        // ✅ JIKA TIDAK ADA KEBUN_ID -> RETURN SUMMARY KESELURUHAN
        if (!$kebun_id) {
            error_log("📊 [LAPORAN-DATA] Fetching OVERALL SUMMARY for user: $current_user_id");

            // Get all kebun milik user
            $stmt_kebun = $conn->prepare("SELECT * FROM data_kebun WHERE user_id = ? ORDER BY nama");
            $stmt_kebun->bind_param("i", $current_user_id);
            $stmt_kebun->execute();
            $result_kebun = $stmt_kebun->get_result();
            $total_kebun = 0;
            $total_luas = 0;

            while ($kebun = $result_kebun->fetch_assoc()) {
                $total_kebun++;
                $total_luas += floatval($kebun['luas']);
            }

            // Get all pemeliharaan data
            $sql_pemeliharaan = "SELECT p.* FROM data_pemeliharaan p 
                             JOIN data_kebun l ON p.kebun_id = l.id 
                             WHERE l.user_id = ?";
            $params = [$current_user_id];
            $types = "i";

            if ($tanggal_dari && $tanggal_sampai) {
                $sql_pemeliharaan .= " AND p.tanggal BETWEEN ? AND ?";
                $params[] = $tanggal_dari;
                $params[] = $tanggal_sampai;
                $types .= "ss";
            }

            $stmt_pemeliharaan = $conn->prepare($sql_pemeliharaan);
            $stmt_pemeliharaan->bind_param($types, ...$params);
            $stmt_pemeliharaan->execute();
            $result_pemeliharaan = $stmt_pemeliharaan->get_result();

            $total_biaya_pemeliharaan = 0;
            $total_pemeliharaan_records = 0;

            while ($row = $result_pemeliharaan->fetch_assoc()) {
                $total_biaya_pemeliharaan += (int)$row['biaya'];
                $total_pemeliharaan_records++;
            }

            // Get all panen data
            $sql_panen = "SELECT p.* FROM data_panen p 
                         JOIN data_kebun l ON p.kebun_id = l.id 
                         WHERE l.user_id = ?";
            $params_panen = [$current_user_id];
            $types_panen = "i";

            if ($tanggal_dari && $tanggal_sampai) {
                $sql_panen .= " AND p.tanggal BETWEEN ? AND ?";
                $params_panen[] = $tanggal_dari;
                $params_panen[] = $tanggal_sampai;
                $types_panen .= "ss";
            }

            $stmt_panen = $conn->prepare($sql_panen);
            $stmt_panen->bind_param($types_panen, ...$params_panen);
            $stmt_panen->execute();
            $result_panen = $stmt_panen->get_result();

            $total_pendapatan = 0;
            $total_panen_records = 0;
            $total_jumlah_kg = 0; // ← TAMBAHAN

            while ($row = $result_panen->fetch_assoc()) {
                $jumlah = (int)$row['jumlah'];
                $harga = (int)$row['harga'];

                $total_jumlah_kg += $jumlah; // ← TAMBAHAN
                $total_pendapatan += $jumlah * $harga;
                $total_panen_records++;
            }

            // Calculate summary
            $total_keuntungan = $total_pendapatan - $total_biaya_pemeliharaan;
            $persentase_keuntungan = $total_biaya_pemeliharaan > 0
                ? round(($total_keuntungan / $total_biaya_pemeliharaan) * 100, 2)
                : 0;

            // ← TAMBAHAN: Hitung rata-rata harga per kg
            $harga_per_kg = $total_jumlah_kg > 0
                ? round($total_pendapatan / $total_jumlah_kg, 2)
                : 0;

            $response_data = [
                "type" => "summary_keseluruhan",
                "total_kebun" => $total_kebun, // ← FIX: Gunakan $total_kebun
                "total_luas" => number_format($total_luas, 1) . " Ha",
                "total_kg" => $total_jumlah_kg,
                "harga_per_kg" => $harga_per_kg,
                "total_biaya_pemeliharaan" => $total_biaya_pemeliharaan,
                "total_pendapatan" => $total_pendapatan,
                "total_keuntungan" => $total_keuntungan,
                "persentase_keuntungan" => $persentase_keuntungan,
                "status_keuntungan" => $total_keuntungan >= 0 ? "untung" : "rugi",
                "total_pemeliharaan_records" => $total_pemeliharaan_records,
                "total_panen_records" => $total_panen_records,
                "periode" => [
                    "tanggal_dari" => $tanggal_dari,
                    "tanggal_sampai" => $tanggal_sampai,
                    "generated_at" => date('Y-m-d H:i:s')
                ]
            ];

            error_log("✅ [LAPORAN-DATA] Overall summary generated: $total_kebun kebun, $total_jumlah_kg kg total, harga/kg: $harga_per_kg");
            sendResponse("success", "Summary keseluruhan berhasil diambil", $response_data);
        }

        // ✅ JIKA ADA KEBUN_ID -> RETURN LAPORAN PER KEBUN (KODE YANG SUDAH ADA)
        error_log("📊 [LAPORAN-DATA] Fetching data for kebun_id: $kebun_id, periode: $tanggal_dari to $tanggal_sampai");

        // Verify kebun ownership & get kebun data
        $stmt_kebun = $conn->prepare("SELECT l.*, u.nama as nama_user 
                                       FROM data_kebun l 
                                       JOIN users u ON l.user_id = u.id 
                                       WHERE l.id = ? AND l.user_id = ?");
        $stmt_kebun->bind_param("ii", $kebun_id, $current_user_id);
        $stmt_kebun->execute();
        $result_kebun = $stmt_kebun->get_result();
        $kebun = $result_kebun->fetch_assoc();

        if (!$kebun) {
            sendResponse("error", "Kebun tidak ditemukan atau bukan milik Anda", null, 404, "KEBUN_NOT_FOUND");
        }

        // Get pemeliharaan data with date filter
        $sql_pemeliharaan = "SELECT p.* 
                          FROM data_pemeliharaan p 
                          JOIN data_kebun l ON p.kebun_id = l.id 
                          WHERE p.kebun_id = ? AND l.user_id = ?";

        $params_pemeliharaan = [$kebun_id, $current_user_id];
        $types_pemeliharaan = "ii";

        if ($tanggal_dari && $tanggal_sampai) {
            $sql_pemeliharaan .= " AND p.tanggal BETWEEN ? AND ?";
            $params_pemeliharaan[] = $tanggal_dari;
            $params_pemeliharaan[] = $tanggal_sampai;
            $types_pemeliharaan .= "ss";
        }

        $sql_pemeliharaan .= " ORDER BY p.tanggal DESC, p.id DESC";

        $stmt_pemeliharaan = $conn->prepare($sql_pemeliharaan);
        $stmt_pemeliharaan->bind_param($types_pemeliharaan, ...$params_pemeliharaan);
        $stmt_pemeliharaan->execute();
        $result_pemeliharaan = $stmt_pemeliharaan->get_result();
        $pemeliharaan_list = [];
        while ($row = $result_pemeliharaan->fetch_assoc()) {
            $pemeliharaan_list[] = [
                "id" => (int)$row['id'],
                "kebun_id" => (int)$row['kebun_id'],
                "kegiatan" => $row['kegiatan'],
                "tanggal" => $row['tanggal'],
                "jumlah" => (int)$row['jumlah'],
                "satuan" => $row['satuan'],
                "biaya" => (int)$row['biaya'],
                "catatan" => $row['catatan']
            ];
        }

        // Get Panen data with date filter
        $sql_panen = "SELECT p.* 
                      FROM data_panen p 
                      JOIN data_kebun l ON p.kebun_id = l.id 
                      WHERE p.kebun_id = ? AND l.user_id = ?";

        $params_panen = [$kebun_id, $current_user_id];
        $types_panen = "ii";

        if ($tanggal_dari && $tanggal_sampai) {
            $sql_panen .= " AND p.tanggal BETWEEN ? AND ?";
            $params_panen[] = $tanggal_dari;
            $params_panen[] = $tanggal_sampai;
            $types_panen .= "ss";
        }

        $sql_panen .= " ORDER BY p.tanggal DESC, p.id DESC";

        $stmt_panen = $conn->prepare($sql_panen);
        $stmt_panen->bind_param($types_panen, ...$params_panen);
        $stmt_panen->execute();
        $result_panen = $stmt_panen->get_result();
        $panen_list = [];

        while ($row = $result_panen->fetch_assoc()) {
            $panen_list[] = [
                "id" => (int)$row['id'],
                "kebun_id" => (int)$row['kebun_id'],
                "tanggal" => $row['tanggal'],
                "jumlah" => (int)$row['jumlah'],
                "harga" => (int)$row['harga'],
                "catatan" => $row['catatan'],
                "total" => (int)$row['jumlah'] * (int)$row['harga']
            ];
        }

        // Calculate summary/statistics
        $total_biaya_pemeliharaan = 0;
        foreach ($pemeliharaan_list as $p) {
            $total_biaya_pemeliharaan += $p['biaya'];
        }

        $total_jumlah_panen = 0;
        $total_pendapatan = 0;
        foreach ($panen_list as $p) {
            $total_jumlah_panen += $p['jumlah'];
            $total_pendapatan += $p['total'];
        }

        $total_keuntungan = $total_pendapatan - $total_biaya_pemeliharaan;
        $persentase_keuntungan = $total_biaya_pemeliharaan > 0
            ? round(($total_keuntungan / $total_biaya_pemeliharaan) * 100, 2)
            : 0;

        // Additional statistics
        $rata_rata_biaya_pemeliharaan = count($pemeliharaan_list) > 0
            ? round($total_biaya_pemeliharaan / count($pemeliharaan_list), 2)
            : 0;

        $rata_rata_panen = count($panen_list) > 0
            ? round($total_jumlah_panen / count($panen_list), 2)
            : 0;

        $harga_rata_rata_per_kg = $total_jumlah_panen > 0
            ? round($total_pendapatan / $total_jumlah_panen, 2)
            : 0;

        // Build response untuk laporan per kebun
        $response_data = [
            "type" => "laporan_kebun",
            "kebun" => [
                "id" => (int)$kebun['id'], // ✅ FIX
                "nama" => $kebun['nama'], // ✅ FIX
                "lokasi" => $kebun['lokasi'], // ✅ FIX
                "luas" => $kebun['luas'], // ✅ FIX
                "titik_tanam" => (int)$kebun['titik_tanam'], // ✅ FIX
                "waktu_beli" => $kebun['waktu_beli'], // ✅ FIX
                "status_kepemilikan" => $kebun['status_kepemilikan'], // ✅ FIX
                "status_kebun" => $kebun['status_kebun'], // ✅ FIX
                "nama_user" => $kebun['nama_user'] // ✅ FIX
            ],
            "periode" => [
                "tanggal_dari" => $tanggal_dari,
                "tanggal_sampai" => $tanggal_sampai,
                "dicetak_pada" => date('Y-m-d H:i:s')
            ],
            "pemeliharaan" => [
                "data" => $pemeliharaan_list,
                "total_records" => count($pemeliharaan_list),
                "total_biaya" => $total_biaya_pemeliharaan,
                "rata_rata_biaya" => $rata_rata_biaya_pemeliharaan
            ],
            "panen" => [
                "data" => $panen_list,
                "total_records" => count($panen_list),
                "total_jumlah_kg" => $total_jumlah_panen,
                "total_pendapatan" => $total_pendapatan,
                "rata_rata_jumlah" => $rata_rata_panen,
                "harga_rata_per_kg" => $harga_rata_rata_per_kg
            ],
            "summary" => [
                "total_biaya_pemeliharaan" => $total_biaya_pemeliharaan,
                "total_pendapatan" => $total_pendapatan,
                "total_keuntungan" => $total_keuntungan,
                "persentase_keuntungan" => $persentase_keuntungan,
                "status_keuntungan" => $total_keuntungan >= 0 ? "untung" : "rugi"
            ],
            "metadata" => [
                "user_id" => $current_user_id,
                "user_name" => $user_payload['nama'],
                "generated_at" => date('Y-m-d H:i:s'),
                "api_version" => "1.0"
            ]
        ];

        error_log("✅ [LAPORAN-DATA] Data fetched successfully: " . count($pemeliharaan_list) . " pemeliharaan, " . count($panen_list) . " panen");

        sendResponse("success", "Data laporan berhasil diambil", $response_data);
    } catch (Exception $e) {
        error_log("💥 [LAPORAN-DATA] Exception: " . $e->getMessage());
        error_log("💥 [LAPORAN-DATA] Stack trace: " . $e->getTraceAsString());
        sendResponse("error", "Terjadi kesalahan sistem: " . $e->getMessage(), null, 500, "SYSTEM_ERROR");
    }
}

sendResponse("error", "Method tidak didukung. Gunakan GET", null, 405, "METHOD_NOT_ALLOWED");
