<?php
error_log("🔍 [PANEN] Request URI: " . $_SERVER['REQUEST_URI']);
error_log("🔍 [PANEN] Query String: " . ($_SERVER['QUERY_STRING'] ?? 'none'));
error_log("🔍 [PANEN] GET params: " . json_encode($_GET));
// filepath: d:\laragon\www\tani-api\panen.php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

// Enable error logging
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/api_debug.log');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/jwt_helper.php';

// Get HTTP method
$method = $_SERVER['REQUEST_METHOD'];

// Function to get input data
function getInputData()
{
    $input = file_get_contents('php://input');
    error_log("📥 [PANEN] Raw input: " . $input);

    $data = json_decode($input, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("❌ [PANEN] JSON decode error: " . json_last_error_msg());
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Invalid JSON format: " . json_last_error_msg(),
            "error_code" => "INVALID_JSON",
            "timestamp" => date('Y-m-d H:i:s')
        ]);
        exit;
    }

    error_log("✅ [PANEN] Parsed data: " . json_encode($data));
    return $data ?? [];
}

// Function untuk response yang konsisten
function sendResponse($status, $message, $data = null, $httpCode = 200, $errorCode = null)
{
    error_log("📤 [PANEN] Sending response: Status=$status, Code=$httpCode, Message=$message");

    http_response_code($httpCode);
    $response = [
        "status" => $status,
        "message" => $message,
        "timestamp" => date('Y-m-d H:i:s')
    ];

    if ($data !== null) {
        $response["data"] = $data;
    }

    if ($errorCode !== null) {
        $response["error_code"] = $errorCode;
    }

    echo json_encode($response);
    exit;
}

// Function to convert date format
function convertDateFormat($dateString)
{
    error_log("📄 [DATE] Converting date: " . $dateString);

    $formats = [
        'Y-m-d',        // 2025-10-03 (already correct)
        'd/m/Y',        // 03/10/2025 (DD/MM/YYYY)
        'm/d/Y',        // 10/03/2025 (MM/DD/YYYY)
        'd-m-Y',        // 03-10-2025 (DD-MM-YYYY)
        'm-d-Y',        // 10-03-2025 (MM-DD-YYYY)
    ];

    foreach ($formats as $format) {
        $date = DateTime::createFromFormat($format, $dateString);
        if ($date && $date->format($format) === $dateString) {
            $converted = $date->format('Y-m-d');
            error_log("✅ [DATE] Successfully converted using format $format: $dateString -> $converted");
            return $converted;
        }
    }

    error_log("❌ [DATE] Failed to convert date: " . $dateString);
    return false;
}

// Get ID from URL if exists
$path = $_SERVER['REQUEST_URI'];
$path_parts = explode('/', trim($path, '/'));
$id = null;

// Extract ID from URL path
foreach ($path_parts as $part) {
    if (is_numeric($part)) {
        $id = (int)$part;
        break;
    }
}

error_log("🔍 [PANEN] Method: $method, Path: $path, ID: " . ($id ?? 'null'));

// ✅ AUTH CHECK - Get current user
try {
    $user_payload = JWTHelper::requireAuth();
    $current_user_id = $user_payload['user_id'];
    error_log("👤 [PANEN] Authenticated user ID: $current_user_id");
} catch (Exception $e) {
    error_log("🚨 [PANEN] Auth failed: " . $e->getMessage());
    sendResponse("error", "Token tidak valid atau telah expire", null, 401, "UNAUTHORIZED");
}

// GET - Read panen
if ($method == 'GET') {
    try {
        // ✅ Check if filtering by kebun_id via query parameter
        $filter_kebun_id = isset($_GET['kebun_id']) ? (int)$_GET['kebun_id'] : null;

        if ($id) {
            // GET /panen/1 - Read by ID (hanya milik user yang login)
            error_log("🔎 [PANEN] Getting panen by ID: $id for user: $current_user_id");

            $sql = "SELECT p.*, l.nama as nama_kebun, l.lokasi as lokasi_kebun, u.nama as nama_user
                    FROM data_panen p 
                    JOIN data_kebun l ON p.kebun_id = l.id 
                    JOIN users u ON l.user_id = u.id
                    WHERE p.id = ? AND l.user_id = ?";
            $stmt = $conn->prepare($sql);

            if (!$stmt) {
                error_log("❌ [PANEN] Prepare failed: " . $conn->error);
                sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
            }

            $stmt->bind_param("ii", $id, $current_user_id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($row = $result->fetch_assoc()) {
                error_log("✅ [PANEN] Found panen ID: " . $row['id']);
                sendResponse("success", "Data panen berhasil ditemukan", $row);
            } else {
                error_log("❌ [PANEN] Panen not found or not owned by user");
                sendResponse("error", "Data panen tidak ditemukan atau bukan milik Anda", null, 404, "PANEN_NOT_FOUND");
            }
        } elseif ($filter_kebun_id) {
            // ✅ GET /panen?kebun_id=1 - Read panen by kebun_id
            error_log("📋 [PANEN] Getting panen for kebun_id: $filter_kebun_id, user: $current_user_id");

            // Validasi kebun_id milik user yang login
            $cek_kebun = $conn->prepare("SELECT nama, lokasi, luas, status_kebun FROM data_kebun WHERE id = ? AND user_id = ?");
            $cek_kebun->bind_param("ii", $filter_kebun_id, $current_user_id);
            $cek_kebun->execute();
            $cek_result = $cek_kebun->get_result();

            if ($cek_result->num_rows == 0) {
                error_log("❌ [PANEN] Kebun not found or not owned by user");
                sendResponse("error", "Kebun tidak ditemukan atau bukan milik Anda", null, 404, "KEBUN_NOT_FOUND");
            }

            $kebun_info = $cek_result->fetch_assoc();

            // Get panen data untuk kebun tersebut
            $sql = "SELECT p.*, l.nama as nama_kebun, l.lokasi as lokasi_kebun, u.nama as nama_user
                    FROM data_panen p 
                    JOIN data_kebun l ON p.kebun_id = l.id 
                    JOIN users u ON l.user_id = u.id
                    WHERE p.kebun_id = ? AND l.user_id = ? 
                    ORDER BY p.tanggal DESC, p.id DESC";
            $stmt = $conn->prepare($sql);

            if (!$stmt) {
                error_log("❌ [PANEN] Prepare failed: " . $conn->error);
                sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
            }

            $stmt->bind_param("ii", $filter_kebun_id, $current_user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $data = [];

            while ($row = $result->fetch_assoc()) {
                $data[] = $row;
            }

            error_log("✅ [PANEN] Found " . count($data) . " panen records for kebun_id: $filter_kebun_id");

            // ✅ Calculate statistics
            $total_kg = 0;
            $total_nilai = 0;
            foreach ($data as $panen) {
                $total_kg += $panen['jumlah'];
                $total_nilai += ($panen['jumlah'] * $panen['harga']);
            }

            $rata_rata_harga = ($total_kg > 0) ? round($total_nilai / $total_kg) : 0;

            $metadata = [
                "total_records" => count($data),
                "kebun_id" => $filter_kebun_id,
                "kebun_info" => $kebun_info,
                "statistics" => [
                    "total_kg" => $total_kg,
                    "total_nilai" => $total_nilai,
                    "rata_rata_harga" => $rata_rata_harga
                ],
                "user_id" => $current_user_id,
                "user_name" => $user_payload['nama']
            ];

            error_log("📊 [PANEN] Statistics - Total Kg: $total_kg, Total Nilai: $total_nilai, Rata-rata Harga: $rata_rata_harga");

            sendResponse("success", "Data panen untuk kebun berhasil diambil", [
                "panen" => $data,
                "metadata" => $metadata
            ]);
        } else {
            // GET /panen - Read semua panen milik user yang login
            error_log("📋 [PANEN] Getting all panen for user: $current_user_id");

            $sql = "SELECT p.*, l.nama as nama_kebun, l.lokasi as lokasi_kebun, u.nama as nama_user
                    FROM data_panen p 
                    JOIN data_kebun l ON p.kebun_id = l.id 
                    JOIN users u ON l.user_id = u.id
                    WHERE l.user_id = ? 
                    ORDER BY p.tanggal DESC, p.id DESC";
            $stmt = $conn->prepare($sql);

            if (!$stmt) {
                error_log("❌ [PANEN] Prepare failed: " . $conn->error);
                sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
            }

            $stmt->bind_param("i", $current_user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $data = [];

            while ($row = $result->fetch_assoc()) {
                $data[] = $row;
            }

            error_log("✅ [PANEN] Found " . count($data) . " panen records");

            // ✅ Calculate statistics for all panen
            $total_kg = 0;
            $total_nilai = 0;
            foreach ($data as $panen) {
                $total_kg += $panen['jumlah'];
                $total_nilai += ($panen['jumlah'] * $panen['harga']);
            }

            $rata_rata_harga = ($total_kg > 0) ? round($total_nilai / $total_kg) : 0;

            $metadata = [
                "total_records" => count($data),
                "statistics" => [
                    "total_kg" => $total_kg,
                    "total_nilai" => $total_nilai,
                    "rata_rata_harga" => $rata_rata_harga
                ],
                "user_id" => $current_user_id,
                "user_name" => $user_payload['nama']
            ];

            error_log("📊 [PANEN] Statistics - Total Kg: $total_kg, Total Nilai: $total_nilai, Rata-rata Harga: $rata_rata_harga");

            sendResponse("success", "Data panen berhasil diambil", [
                "panen" => $data,
                "metadata" => $metadata
            ]);
        }
    } catch (Exception $e) {
        error_log("💥 [PANEN] GET Exception: " . $e->getMessage());
        sendResponse("error", "Terjadi kesalahan sistem", null, 500, "SYSTEM_ERROR");
    }
}


// POST - Create new panen
if ($method == 'POST') {
    try {
        error_log("🌱 [PANEN] Creating new panen for user: $current_user_id");

        $input = getInputData();

        // Validation
        $required_fields = ['kebun_id', 'tanggal', 'jumlah', 'harga'];
        $missing_fields = [];

        foreach ($required_fields as $field) {
            if (!isset($input[$field]) || $input[$field] === '' || $input[$field] === null) {
                $missing_fields[] = $field;
            }
        }

        if (!empty($missing_fields)) {
            error_log("❌ [PANEN] Missing fields: " . implode(', ', $missing_fields));
            sendResponse("error", "Field berikut wajib diisi: " . implode(', ', $missing_fields), null, 400, "MISSING_REQUIRED_FIELDS");
        }

        $kebun_id = (int)$input['kebun_id'];
        $tanggal_raw = trim($input['tanggal']);
        $jumlah = (int)$input['jumlah'];
        $harga = (int)$input['harga'];
        $catatan = isset($input['catatan']) ? trim($input['catatan']) : null;

        // Convert date format
        $tanggal = convertDateFormat($tanggal_raw);
        if (!$tanggal) {
            error_log("❌ [PANEN] Invalid date format: " . $tanggal_raw);
            sendResponse("error", "Format tanggal tidak valid. Gunakan format DD/MM/YYYY atau YYYY-MM-DD", null, 400, "INVALID_DATE_FORMAT");
        }

        error_log("✅ [PANEN] Date converted: $tanggal_raw -> $tanggal");

        // Validasi kebun_id milik user yang login
        $cek_kebun = $conn->prepare("SELECT nama, lokasi FROM data_kebun WHERE id = ? AND user_id = ?");
        $cek_kebun->bind_param("ii", $kebun_id, $current_user_id);
        $cek_kebun->execute();
        $cek_result = $cek_kebun->get_result();
        if ($cek_result->num_rows == 0) {
            error_log("❌ [PANEN] Kebun not found or not owned by user");
            sendResponse("error", "Kebun tidak ditemukan atau bukan milik Anda", null, 404, "KEBUN_NOT_FOUND");
        }

        $kebun_data = $cek_result->fetch_assoc();

        // ✅ NEW: Check monthly limit (maksimal 2 data panen per bulan per kebun)
        $tahun_bulan = date('Y-m', strtotime($tanggal));
        error_log("🔍 [PANEN] Checking monthly limit for kebun: $kebun_id, month: $tahun_bulan");

        // ✅ PERBAIKAN: Spesifikkan nama tabel untuk kolom id
        $cek_limit = $conn->prepare("SELECT COUNT(*) as jumlah, 
                            GROUP_CONCAT(CONCAT('ID:', p.id, ' (', p.tanggal, ')') SEPARATOR ', ') as data_existing
                            FROM data_panen p 
                            JOIN data_kebun l ON p.kebun_id = l.id 
                            WHERE p.kebun_id = ? 
                            AND l.user_id = ? 
                            AND DATE_FORMAT(p.tanggal, '%Y-%m') = ?");
        $cek_limit->bind_param("iis", $kebun_id, $current_user_id, $tahun_bulan);
        $cek_limit->execute();
        $limit_result = $cek_limit->get_result();
        $limit_data = $limit_result->fetch_assoc();
        $jumlah_panen_bulan_ini = (int)$limit_data['jumlah'];
        $data_existing = $limit_data['data_existing'];

        error_log("📊 [PANEN] Current month data count: $jumlah_panen_bulan_ini");
        error_log("📊 [PANEN] Existing data: " . ($data_existing ?: 'none'));

        if ($jumlah_panen_bulan_ini >= 2) {
            error_log("❌ [PANEN] Monthly limit exceeded: $jumlah_panen_bulan_ini/2");

            sendResponse(
                "error",
                "Maksimal 2 data panen per bulan per kebun. " .
                    "Bulan $tahun_bulan sudah memiliki $jumlah_panen_bulan_ini data. " .
                    "Hapus salah satu data berikut untuk menambah data baru: $data_existing",
                [
                    "limit_info" => [
                        "maksimal_per_bulan" => 2,
                        "jumlah_saat_ini" => $jumlah_panen_bulan_ini,
                        "bulan" => $tahun_bulan,
                        "kebun_id" => $kebun_id,
                        "nama_kebun" => $kebun_data['nama'],
                        "data_existing" => $data_existing
                    ]
                ],
                400,
                "MONTHLY_LIMIT_EXCEEDED"
            );
        }

        // Additional validation
        if ($jumlah <= 0) {
            sendResponse("error", "Jumlah panen harus lebih dari 0", null, 400, "INVALID_JUMLAH");
        }

        if ($harga < 0) {
            sendResponse("error", "Harga tidak boleh negatif", null, 400, "INVALID_HARGA");
        }

        error_log("✅ [PANEN] Data validated successfully");
        error_log("✅ [PANEN] Monthly limit check passed: $jumlah_panen_bulan_ini/2");
        error_log("📊 [PANEN] Data to insert: kebun_id=$kebun_id, tanggal=$tanggal, jumlah=$jumlah, harga=$harga");

        // Insert data panen
        $sql = "INSERT INTO data_panen (kebun_id, tanggal, jumlah, harga, catatan) VALUES (?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            error_log("❌ [PANEN] Prepare failed: " . $conn->error);
            sendResponse("error", "Database prepare error: " . $conn->error, null, 500, "DATABASE_PREPARE_ERROR");
        }

        $stmt->bind_param("isiis", $kebun_id, $tanggal, $jumlah, $harga, $catatan);
        $result = $stmt->execute();

        if (!$result) {
            error_log("❌ [PANEN] Execute failed: " . $stmt->error);
            sendResponse("error", "Database execute error: " . $stmt->error, null, 500, "DATABASE_EXECUTE_ERROR");
        }

        if ($result) {
            $new_id = $conn->insert_id;
            error_log("✅ [PANEN] Successfully created panen with ID: $new_id");
            error_log("📊 [PANEN] Monthly count after insert: " . ($jumlah_panen_bulan_ini + 1) . "/2");

            $response_data = [
                "id" => $new_id,
                "kebun_id" => $kebun_id,
                "tanggal" => $tanggal,
                "jumlah" => $jumlah,
                "harga" => $harga,
                "catatan" => $catatan,
                "nama_kebun" => $kebun_data['nama'],
                "lokasi_kebun" => $kebun_data['lokasi'],
                "nama_user" => $user_payload['nama'],
                "limit_info" => [
                    "bulan" => $tahun_bulan,
                    "jumlah_setelah_insert" => $jumlah_panen_bulan_ini + 1,
                    "maksimal_per_bulan" => 2,
                    "sisa_slot" => 2 - ($jumlah_panen_bulan_ini + 1)
                ]
            ];

            sendResponse("success", "Data panen berhasil ditambahkan", $response_data, 201);
        } else {
            error_log("❌ [PANEN] Insert failed but no error message");
            sendResponse("error", "Gagal menambahkan data panen", null, 500, "DATABASE_ERROR");
        }
    } catch (Exception $e) {
        error_log("💥 [PANEN] POST Exception: " . $e->getMessage());
        error_log("💥 [PANEN] Stack trace: " . $e->getTraceAsString());
        sendResponse("error", "Terjadi kesalahan sistem: " . $e->getMessage(), null, 500, "SYSTEM_ERROR");
    }
}

// PUT - Update panen
if ($method == 'PUT') {
    try {
        if (!$id) {
            sendResponse("error", "ID panen diperlukan untuk update", null, 400, "MISSING_ID");
        }

        error_log("📝 [PANEN] Updating panen ID: $id for user: $current_user_id");

        $input = getInputData();

        // Validation
        $required_fields = ['kebun_id', 'tanggal', 'jumlah', 'harga'];
        $missing_fields = [];

        foreach ($required_fields as $field) {
            if (!isset($input[$field]) || $input[$field] === '' || $input[$field] === null) {
                $missing_fields[] = $field;
            }
        }

        if (!empty($missing_fields)) {
            sendResponse("error", "Field berikut wajib diisi: " . implode(', ', $missing_fields), null, 400, "MISSING_REQUIRED_FIELDS");
        }

        // Cek apakah panen exists dan milik user yang login
        $cek = $conn->prepare("SELECT p.id FROM data_panen p JOIN data_kebun l ON p.kebun_id = l.id WHERE p.id = ? AND l.user_id = ?");
        $cek->bind_param("ii", $id, $current_user_id);
        $cek->execute();
        $cek->store_result();

        if ($cek->num_rows == 0) {
            sendResponse("error", "Data panen tidak ditemukan atau bukan milik Anda", null, 404, "PANEN_NOT_FOUND");
        }

        $kebun_id = (int)$input['kebun_id'];
        $tanggal_raw = trim($input['tanggal']);
        $jumlah = (int)$input['jumlah'];
        $harga = (int)$input['harga'];
        $catatan = isset($input['catatan']) ? trim($input['catatan']) : null;

        // Convert date format
        $tanggal = convertDateFormat($tanggal_raw);
        if (!$tanggal) {
            sendResponse("error", "Format tanggal tidak valid. Gunakan format DD/MM/YYYY atau YYYY-MM-DD", null, 400, "INVALID_DATE_FORMAT");
        }

        // Validasi kebun_id milik user yang login
        $cek_kebun = $conn->prepare("SELECT nama, lokasi FROM data_kebun WHERE id = ? AND user_id = ?");
        $cek_kebun->bind_param("ii", $kebun_id, $current_user_id);
        $cek_kebun->execute();
        $cek_result = $cek_kebun->get_result();
        if ($cek_result->num_rows == 0) {
            sendResponse("error", "Kebun tidak ditemukan atau bukan milik Anda", null, 404, "KEBUN_NOT_FOUND");
        }

        $kebun_data = $cek_result->fetch_assoc();

        // Additional validation
        if ($jumlah <= 0) {
            sendResponse("error", "Jumlah panen harus lebih dari 0", null, 400, "INVALID_JUMLAH");
        }

        if ($harga < 0) {
            sendResponse("error", "Harga tidak boleh negatif", null, 400, "INVALID_HARGA");
        }

        // Update data panen
        $sql = "UPDATE data_panen p 
                JOIN data_kebun l ON p.kebun_id = l.id 
                SET p.kebun_id = ?, p.tanggal = ?, p.jumlah = ?, p.harga = ?, p.catatan = ? 
                WHERE p.id = ? AND l.user_id = ?";
        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            error_log("❌ [PANEN] Update prepare failed: " . $conn->error);
            sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
        }

        $stmt->bind_param("isiisii", $kebun_id, $tanggal, $jumlah, $harga, $catatan, $id, $current_user_id);
        $result = $stmt->execute();

        if ($result && $stmt->affected_rows > 0) {
            error_log("✅ [PANEN] Successfully updated panen ID: $id");

            $response_data = [
                "id" => $id,
                "kebun_id" => $kebun_id,
                "tanggal" => $tanggal,
                "jumlah" => $jumlah,
                "harga" => $harga,
                "catatan" => $catatan,
                "nama_kebun" => $kebun_data['nama'],
                "lokasi_kebun" => $kebun_data['lokasi'],
                "nama_user" => $user_payload['nama']
            ];

            sendResponse("success", "Data panen berhasil diupdate", $response_data);
        } else {
            error_log("❌ [PANEN] Update failed or no rows affected");
            sendResponse("error", "Data panen tidak ditemukan atau bukan milik Anda", null, 404, "PANEN_NOT_FOUND");
        }
    } catch (Exception $e) {
        error_log("💥 [PANEN] PUT Exception: " . $e->getMessage());
        sendResponse("error", "Terjadi kesalahan sistem", null, 500, "SYSTEM_ERROR");
    }
}

// DELETE - Delete panen
if ($method == 'DELETE') {
    try {
        if (!$id) {
            sendResponse("error", "ID panen diperlukan untuk delete", null, 400, "MISSING_ID");
        }

        error_log("🗑️ [PANEN] Deleting panen ID: $id for user: $current_user_id");

        // Cek apakah panen exists dan milik user yang login
        $cek = $conn->prepare("SELECT p.id FROM data_panen p JOIN data_kebun l ON p.kebun_id = l.id WHERE p.id = ? AND l.user_id = ?");
        $cek->bind_param("ii", $id, $current_user_id);
        $cek->execute();
        $cek->store_result();

        if ($cek->num_rows == 0) {
            sendResponse("error", "Data panen tidak ditemukan atau bukan milik Anda", null, 404, "PANEN_NOT_FOUND");
        }

        // Hapus panen
        $sql = "DELETE p FROM data_panen p JOIN data_kebun l ON p.kebun_id = l.id WHERE p.id = ? AND l.user_id = ?";
        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            error_log("❌ [PANEN] Delete prepare failed: " . $conn->error);
            sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
        }

        $stmt->bind_param("ii", $id, $current_user_id);
        $result = $stmt->execute();

        if ($result && $stmt->affected_rows > 0) {
            error_log("✅ [PANEN] Successfully deleted panen ID: $id");
            sendResponse("success", "Data panen berhasil dihapus");
        } else {
            error_log("❌ [PANEN] Delete failed or no rows affected");
            sendResponse("error", "Data panen tidak ditemukan atau bukan milik Anda", null, 404, "PANEN_NOT_FOUND");
        }
    } catch (Exception $e) {
        error_log("💥 [PANEN] DELETE Exception: " . $e->getMessage());
        sendResponse("error", "Terjadi kesalahan sistem", null, 500, "SYSTEM_ERROR");
    }
}

// Method not allowed
error_log("❌ [PANEN] Method not allowed: $method");
sendResponse("error", "Method tidak didukung", null, 405, "METHOD_NOT_ALLOWED");
