<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

// Enable error logging
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/api_debug.log');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/jwt_helper.php';

// Get HTTP method
$method = $_SERVER['REQUEST_METHOD'];

// Function to get input data
function getInputData()
{
    $input = file_get_contents('php://input');
    error_log("📥 [PEMELIHARAAN] Raw input: " . $input);

    $data = json_decode($input, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("❌ [PEMELIHARAAN] JSON decode error: " . json_last_error_msg());
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Invalid JSON format: " . json_last_error_msg(),
            "error_code" => "INVALID_JSON",
            "timestamp" => date('Y-m-d H:i:s')
        ]);
        exit;
    }

    error_log("✅ [PEMELIHARAAN] Parsed data: " . json_encode($data));
    return $data ?? [];
}

// Function untuk response yang konsisten
function sendResponse($status, $message, $data = null, $httpCode = 200, $errorCode = null)
{
    error_log("📤 [PEMELIHARAAN] Sending response: Status=$status, Code=$httpCode, Message=$message");

    http_response_code($httpCode);
    $response = [
        "status" => $status,
        "message" => $message,
        "timestamp" => date('Y-m-d H:i:s')
    ];

    if ($data !== null) {
        $response["data"] = $data;
    }

    if ($errorCode !== null) {
        $response["error_code"] = $errorCode;
    }

    echo json_encode($response);
    exit;
}

// Function to convert date format
function convertDateFormat($dateString)
{
    error_log("🔄 [DATE] Converting date: " . $dateString);

    $formats = [
        'Y-m-d',        // 2025-10-03 (already correct)
        'd/m/Y',        // 03/10/2025 (DD/MM/YYYY)
        'm/d/Y',        // 10/03/2025 (MM/DD/YYYY)
        'd-m-Y',        // 03-10-2025 (DD-MM-YYYY)
        'm-d-Y',        // 10-03-2025 (MM-DD-YYYY)
    ];

    foreach ($formats as $format) {
        $date = DateTime::createFromFormat($format, $dateString);
        if ($date && $date->format($format) === $dateString) {
            $converted = $date->format('Y-m-d');
            error_log("✅ [DATE] Successfully converted using format $format: $dateString -> $converted");
            return $converted;
        }
    }

    error_log("❌ [DATE] Failed to convert date: " . $dateString);
    return false;
}

// Get ID from URL if exists
$path = $_SERVER['REQUEST_URI'];
$path_parts = explode('/', trim($path, '/'));
$id = null;

// Extract ID from URL path
foreach ($path_parts as $part) {
    if (is_numeric($part)) {
        $id = (int)$part;
        break;
    }
}

error_log("🔍 [PEMELIHARAAN] Method: $method, Path: $path, ID: " . ($id ?? 'null'));

// ✅ AUTH CHECK - Get current user
try {
    $user_payload = JWTHelper::requireAuth();
    $current_user_id = $user_payload['user_id'];
    error_log("👤 [PEMELIHARAAN] Authenticated user ID: $current_user_id");
} catch (Exception $e) {
    error_log("🚨 [PEMELIHARAAN] Auth failed: " . $e->getMessage());
    sendResponse("error", "Token tidak valid atau telah expire", null, 401, "UNAUTHORIZED");
}
// GET - Read operations
if ($method == 'GET') {
    try {
        // ✅ GET /pemeliharaan?kegiatan_list=true - Return list of allowed kegiatan
        if (isset($_GET['kegiatan_list']) && $_GET['kegiatan_list'] == 'true') {
            error_log("📋 [PEMELIHARAAN] Getting kegiatan list");

            $allowed_kegiatan = [
                'Pemupukan',
                'Penyemprotan',
                'Pemangkasan Tunas',
                'Pemangkasan Gulma',
                'Penyiraman'
            ];

            sendResponse("success", "Daftar kegiatan pemeliharaan tersedia", [
                "kegiatan_list" => $allowed_kegiatan,
                "total" => count($allowed_kegiatan)
            ]);
        }

        // Check query parameters for filtering
        $kebun_id_param = isset($_GET['kebun_id']) ? (int)$_GET['kebun_id'] : null;
        $kegiatan_param = isset($_GET['kegiatan']) ? trim($_GET['kegiatan']) : null;

        if ($kebun_id_param && $kegiatan_param) {
            // ✅ GET /pemeliharaan?kebun_id=2&kegiatan=Pemupukan
            error_log("🔍 [PEMELIHARAAN] Getting pemeliharaan by kebun_id: $kebun_id_param AND kegiatan: $kegiatan_param");

            // Validasi kebun
            $cek_kebun = $conn->prepare("SELECT nama, lokasi FROM data_kebun WHERE id = ? AND user_id = ?");
            $cek_kebun->bind_param("ii", $kebun_id_param, $current_user_id);
            $cek_kebun->execute();
            $cek_result = $cek_kebun->get_result();

            if ($cek_result->num_rows == 0) {
                sendResponse("error", "Kebun tidak ditemukan atau bukan milik Anda", null, 404, "KEBUN_NOT_FOUND");
            }

            $kebun_data = $cek_result->fetch_assoc();

            // Get pemeliharaan
            $sql = "SELECT p.*, l.nama as nama_kebun, l.lokasi as lokasi_kebun, u.nama as nama_user
            FROM data_pemeliharaan p 
            JOIN data_kebun l ON p.kebun_id = l.id 
            JOIN users u ON l.user_id = u.id
            WHERE p.kebun_id = ? AND p.kegiatan = ? AND l.user_id = ? 
            ORDER BY p.tanggal DESC, p.id DESC";
            $stmt = $conn->prepare($sql);

            if (!$stmt) {
                sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
            }

            $stmt->bind_param("isi", $kebun_id_param, $kegiatan_param, $current_user_id);
            $stmt->execute();
            $result = $stmt->get_result();

            $data = [];
            $total_biaya = 0;

            while ($row = $result->fetch_assoc()) {
                $data[] = $row;
                if (isset($row['biaya'])) {
                    $total_biaya += (int)$row['biaya'];
                }
            }

            error_log("✅ [PEMELIHARAAN] Found " . count($data) . " records, total biaya: " . $total_biaya);

            // ✅ FIX: Metadata dengan struktur BENAR
            $metadata = [
                "total_records" => count($data),
                "statistics" => [
                    "total_biaya" => $total_biaya,
                    "rata_rata_biaya" => count($data) > 0
                        ? (int)round($total_biaya / count($data))
                        : 0, // ✅ Pastikan selalu integer
                    "kegiatan_terbanyak" => $kegiatan_param
                ],
                "kebun_id" => $kebun_id_param,
                "nama_kebun" => $kebun_data['nama'],
                "lokasi_kebun" => $kebun_data['lokasi'],
                "kegiatan_filter" => $kegiatan_param,
                "user_id" => $current_user_id,
                "user_name" => $user_payload['nama']
            ];

            error_log("📤 [PEMELIHARAAN] Sending metadata: " . json_encode($metadata));

            sendResponse("success", "Data pemeliharaan berhasil diambil", [
                "pemeliharaan" => $data,
                "metadata" => $metadata
            ]);
        } elseif ($kebun_id_param) {
            // ✅ GET /pemeliharaan?kebun_id=2 - Filter by kebun_id only
            error_log("🏡 [PEMELIHARAAN] Getting pemeliharaan by kebun_id: $kebun_id_param for user: $current_user_id");

            // Validasi kebun milik user
            $cek_kebun = $conn->prepare("SELECT nama, lokasi FROM data_kebun WHERE id = ? AND user_id = ?");
            $cek_kebun->bind_param("ii", $kebun_id_param, $current_user_id);
            $cek_kebun->execute();
            $cek_result = $cek_kebun->get_result();

            if ($cek_result->num_rows == 0) {
                error_log("❌ [PEMELIHARAAN] Kebun not found or not owned by user");
                sendResponse("error", "Kebun tidak ditemukan atau bukan milik Anda", null, 404, "KEBUN_NOT_FOUND");
            }

            $kebun_data = $cek_result->fetch_assoc();

            // Get all pemeliharaan for this kebun
            $sql = "SELECT p.*, l.nama as nama_kebun, l.lokasi as lokasi_kebun, u.nama as nama_user
            FROM data_pemeliharaan p 
            JOIN data_kebun l ON p.kebun_id = l.id 
            JOIN users u ON l.user_id = u.id
            WHERE p.kebun_id = ? AND l.user_id = ? 
            ORDER BY p.tanggal DESC, p.id DESC";
            $stmt = $conn->prepare($sql);

            if (!$stmt) {
                error_log("❌ [PEMELIHARAAN] Prepare failed: " . $conn->error);
                sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
            }

            $stmt->bind_param("ii", $kebun_id_param, $current_user_id);
            $stmt->execute();
            $result = $stmt->get_result();

            $data = [];
            $total_biaya = 0;
            $kegiatan_count = []; // ✅ TAMBAH

            while ($row = $result->fetch_assoc()) {
                $data[] = $row;

                if (isset($row['biaya'])) {
                    $total_biaya += (int)$row['biaya'];
                }

                // ✅ TAMBAH: Hitung kegiatan
                $kegiatan = $row['kegiatan'];
                if (!isset($kegiatan_count[$kegiatan])) {
                    $kegiatan_count[$kegiatan] = 0;
                }
                $kegiatan_count[$kegiatan]++;
            }

            // ✅ TAMBAH: Cari kegiatan terbanyak
            $kegiatan_terbanyak = null;
            $max_count = 0;
            foreach ($kegiatan_count as $kegiatan => $count) {
                if ($count > $max_count) {
                    $max_count = $count;
                    $kegiatan_terbanyak = $kegiatan;
                }
            }

            error_log("✅ [PEMELIHARAAN] Found " . count($data) . " pemeliharaan records for kebun_id: $kebun_id_param with total biaya: " . $total_biaya);

            // ✅ FIX: Metadata dengan struktur BENAR
            $metadata = [
                "total_records" => count($data),
                "statistics" => [
                    "total_biaya" => $total_biaya,
                    "rata_rata_biaya" => count($data) > 0
                        ? (int)round($total_biaya / count($data)) // ✅ Ubah ke (int)
                        : 0, // ✅ Pastikan selalu integer
                    "kegiatan_terbanyak" => $kegiatan_terbanyak
                ],
                "kebun_id" => $kebun_id_param,
                "nama_kebun" => $kebun_data['nama'],
                "lokasi_kebun" => $kebun_data['lokasi'],
                "user_id" => $current_user_id,
                "user_name" => $user_payload['nama']
            ];

            error_log("📤 [PEMELIHARAAN] Sending metadata: " . json_encode($metadata));

            sendResponse("success", "Data pemeliharaan berhasil diambil untuk kebun " . $kebun_data['nama'], [
                "pemeliharaan" => $data,
                "metadata" => $metadata
            ]);
        } elseif ($kegiatan_param) {
            // ✅ GET /pemeliharaan?kegiatan=Pemupukan - Filter by kegiatan only
            error_log("🔍 [PEMELIHARAAN] Getting pemeliharaan by kegiatan: $kegiatan_param for user: $current_user_id");

            $sql = "SELECT p.*, l.nama as nama_kebun, l.lokasi as lokasi_kebun, u.nama as nama_user
            FROM data_pemeliharaan p 
            JOIN data_kebun l ON p.kebun_id = l.id 
            JOIN users u ON l.user_id = u.id
            WHERE p.kegiatan = ? AND l.user_id = ? 
            ORDER BY p.tanggal DESC, p.id DESC";
            $stmt = $conn->prepare($sql);

            if (!$stmt) {
                error_log("❌ [PEMELIHARAAN] Prepare failed: " . $conn->error);
                sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
            }

            $stmt->bind_param("si", $kegiatan_param, $current_user_id);
            $stmt->execute();
            $result = $stmt->get_result();

            $data = [];
            $total_biaya = 0;

            while ($row = $result->fetch_assoc()) {
                $data[] = $row;
                if (isset($row['biaya'])) {
                    $total_biaya += (int)$row['biaya'];
                }
            }

            error_log("✅ [PEMELIHARAAN] Found " . count($data) . " pemeliharaan records for kegiatan: $kegiatan_param with total biaya: " . $total_biaya);

            // ✅ FIX: Metadata dengan struktur BENAR
            $metadata = [
                "total_records" => count($data),
                "statistics" => [
                    "total_biaya" => $total_biaya,
                    "rata_rata_biaya" => count($data) > 0
                        ? (int)round($total_biaya / count($data)) // ✅ Ubah ke (int)
                        : 0,
                    "kegiatan_terbanyak" => $kegiatan_param
                ],
                "kegiatan_filter" => $kegiatan_param, // ✅ Tandai sedang filter
                "user_id" => $current_user_id,
                "user_name" => $user_payload['nama']
            ];

            error_log("📤 [PEMELIHARAAN] Sending metadata: " . json_encode($metadata));

            sendResponse("success", "Data pemeliharaan berhasil diambil untuk kegiatan " . $kegiatan_param, [
                "pemeliharaan" => $data,
                "metadata" => $metadata
            ]);
        } elseif ($id) {
            // GET /pemeliharaan/1 - Read by ID (hanya milik user yang login)
            error_log("🔎 [PEMELIHARAAN] Getting pemeliharaan by ID: $id for user: $current_user_id");

            $sql = "SELECT p.*, l.nama as nama_kebun, l.lokasi as lokasi_kebun, u.nama as nama_user
                    FROM data_pemeliharaan p 
                    JOIN data_kebun l ON p.kebun_id = l.id 
                    JOIN users u ON l.user_id = u.id
                    WHERE p.id = ? AND l.user_id = ?";
            $stmt = $conn->prepare($sql);

            if (!$stmt) {
                error_log("❌ [PEMELIHARAAN] Prepare failed: " . $conn->error);
                sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
            }

            $stmt->bind_param("ii", $id, $current_user_id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($row = $result->fetch_assoc()) {
                error_log("✅ [PEMELIHARAAN] Found pemeliharaan ID: " . $row['id']);
                sendResponse("success", "Data pemeliharaan berhasil ditemukan", $row);
            } else {
                error_log("❌ [PEMELIHARAAN] Pemeliharaan not found or not owned by user");
                sendResponse("error", "Data pemeliharaan tidak ditemukan atau bukan milik Anda", null, 404, "PEMELIHARAAN_NOT_FOUND");
            }
        } else {
            error_log("📋 [PEMELIHARAAN] Getting all pemeliharaan for user: $current_user_id");

            $sql = "SELECT p.*, l.nama as nama_kebun, l.lokasi as lokasi_kebun, u.nama as nama_user
            FROM data_pemeliharaan p 
            JOIN data_kebun l ON p.kebun_id = l.id 
            JOIN users u ON l.user_id = u.id
            WHERE l.user_id = ? 
            ORDER BY p.tanggal DESC, p.id DESC";
            $stmt = $conn->prepare($sql);

            if (!$stmt) {
                error_log("❌ [PEMELIHARAAN] Prepare failed: " . $conn->error);
                sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
            }

            $stmt->bind_param("i", $current_user_id);
            $stmt->execute();
            $result = $stmt->get_result();

            $data = [];
            $total_biaya = 0;
            $kegiatan_count = []; // ✅ TAMBAH: Hitung kegiatan terbanyak

            while ($row = $result->fetch_assoc()) {
                $data[] = $row;

                // ✅ Sum total biaya
                if (isset($row['biaya'])) {
                    $total_biaya += (int)$row['biaya']; // ✅ Gunakan int, bukan float
                }

                // ✅ TAMBAH: Hitung kegiatan
                $kegiatan = $row['kegiatan'];
                if (!isset($kegiatan_count[$kegiatan])) {
                    $kegiatan_count[$kegiatan] = 0;
                }
                $kegiatan_count[$kegiatan]++;
            }

            // ✅ TAMBAH: Cari kegiatan terbanyak
            $kegiatan_terbanyak = null;
            $max_count = 0;
            foreach ($kegiatan_count as $kegiatan => $count) {
                if ($count > $max_count) {
                    $max_count = $count;
                    $kegiatan_terbanyak = $kegiatan;
                }
            }

            error_log("✅ [PEMELIHARAAN] Found " . count($data) . " pemeliharaan records with total biaya: " . $total_biaya);
            error_log("📊 [PEMELIHARAAN] Kegiatan terbanyak: " . ($kegiatan_terbanyak ?? 'null'));

            // ✅ FIX: Metadata dengan struktur BENAR (sesuai Flutter)
            $metadata = [
                "total_records" => count($data),
                "statistics" => [
                    "total_biaya" => $total_biaya,
                    "rata_rata_biaya" => count($data) > 0
                        ? (int)round($total_biaya / count($data)) // ✅ Ubah ke (int)
                        : 0,
                    "kegiatan_terbanyak" => $kegiatan_terbanyak
                ],
                "user_id" => $current_user_id,
                "user_name" => $user_payload['nama']
            ];

            // ✅ ADD: Debug log untuk verify data
            error_log("📤 [PEMELIHARAAN] Sending metadata: " . json_encode($metadata));

            sendResponse("success", "Data pemeliharaan berhasil diambil", [
                "pemeliharaan" => $data,
                "metadata" => $metadata
            ]);
        }
    } catch (Exception $e) {
        error_log("💥 [PEMELIHARAAN] GET Exception: " . $e->getMessage());
        sendResponse("error", "Terjadi kesalahan sistem", null, 500, "SYSTEM_ERROR");
    }
}

// POST - Create new pemeliharaan
if ($method == 'POST') {
    try {
        error_log("📝 [PEMELIHARAAN] Creating new pemeliharaan for user: $current_user_id");

        $input = getInputData();

        // Validation
        $required_fields = ['kebun_id', 'kegiatan', 'tanggal', 'jumlah', 'biaya'];
        $missing_fields = [];

        foreach ($required_fields as $field) {
            if (!isset($input[$field]) || $input[$field] === '' || $input[$field] === null) {
                $missing_fields[] = $field;
            }
        }

        if (!empty($missing_fields)) {
            error_log("❌ [PEMELIHARAAN] Missing fields: " . implode(', ', $missing_fields));
            sendResponse("error", "Field berikut wajib diisi: " . implode(', ', $missing_fields), null, 400, "MISSING_REQUIRED_FIELDS");
        }

        // Ambil data input
        $kebun_id = (int)$input['kebun_id'];
        $kegiatan = trim($input['kegiatan']);
        $tanggal_raw = trim($input['tanggal']);
        $jumlah = (int)$input['jumlah'];
        $satuan = isset($input['satuan']) ? trim($input['satuan']) : null;
        $biaya = (int)$input['biaya'];
        $catatan = isset($input['catatan']) ? trim($input['catatan']) : null;

        error_log("📋 [PEMELIHARAAN] Input data - Kebun: $kebun_id, Kegiatan: $kegiatan, Tanggal: $tanggal_raw");

        // Convert date format DD-MM-YYYY to YYYY-MM-DD
        $tanggal = convertDateFormat($tanggal_raw);
        if (!$tanggal) {
            error_log("❌ [PEMELIHARAAN] Invalid date format: $tanggal_raw");
            sendResponse("error", "Format tanggal tidak valid. Gunakan format DD-MM-YYYY atau YYYY-MM-DD", null, 400, "INVALID_DATE_FORMAT");
        }

        // Validasi kebun_id milik user yang login
        $cek_kebun = $conn->prepare("SELECT nama, lokasi FROM data_kebun WHERE id = ? AND user_id = ?");
        $cek_kebun->bind_param("ii", $kebun_id, $current_user_id);
        $cek_kebun->execute();
        $cek_result = $cek_kebun->get_result();

        if ($cek_result->num_rows == 0) {
            error_log("❌ [PEMELIHARAAN] Kebun not found or not owned by user");
            sendResponse("error", "Kebun tidak ditemukan atau bukan milik Anda", null, 404, "KEBUN_NOT_FOUND");
        }

        $kebun_data = $cek_result->fetch_assoc();

        // Additional validation
        if (strlen($kegiatan) < 3) {
            sendResponse("error", "Kegiatan minimal 3 karakter", null, 400, "INVALID_KEGIATAN");
        }

        if ($jumlah <= 0) {
            sendResponse("error", "Jumlah harus lebih dari 0", null, 400, "INVALID_JUMLAH");
        }

        if ($biaya < 0) {
            sendResponse("error", "Biaya tidak boleh negatif", null, 400, "INVALID_BIAYA");
        }

        // ✅ INSERT INTO DATABASE
        $sql = "INSERT INTO data_pemeliharaan (kebun_id, kegiatan, tanggal, jumlah, satuan, biaya, catatan) 
                VALUES (?, ?, ?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            error_log("❌ [PEMELIHARAAN] Prepare failed: " . $conn->error);
            sendResponse("error", "Database error: " . $conn->error, null, 500, "DATABASE_ERROR");
        }

        $stmt->bind_param("issisis", $kebun_id, $kegiatan, $tanggal, $jumlah, $satuan, $biaya, $catatan);
        $result = $stmt->execute();

        if ($result) {
            $new_id = $conn->insert_id;
            error_log("✅ [PEMELIHARAAN] Successfully created pemeliharaan ID: $new_id");

            $response_data = [
                "id" => $new_id,
                "kebun_id" => $kebun_id,
                "kegiatan" => $kegiatan,
                "tanggal" => $tanggal,
                "jumlah" => $jumlah,
                "satuan" => $satuan,
                "biaya" => $biaya,
                "catatan" => $catatan,
                "nama_kebun" => $kebun_data['nama'],
                "lokasi_kebun" => $kebun_data['lokasi'],
                "nama_user" => $user_payload['nama']
            ];

            sendResponse("success", "Data pemeliharaan berhasil ditambahkan", $response_data, 201);
        } else {
            error_log("❌ [PEMELIHARAAN] Insert failed: " . $stmt->error);
            sendResponse("error", "Gagal menambahkan data pemeliharaan: " . $stmt->error, null, 500, "INSERT_FAILED");
        }
    } catch (Exception $e) {
        error_log("💥 [PEMELIHARAAN] POST Exception: " . $e->getMessage());
        sendResponse("error", "Terjadi kesalahan sistem: " . $e->getMessage(), null, 500, "SYSTEM_ERROR");
    }
}

// PUT - Update pemeliharaan
if ($method == 'PUT') {
    try {
        if (!$id) {
            sendResponse("error", "ID pemeliharaan diperlukan untuk update", null, 400, "MISSING_ID");
        }

        error_log("📝 [PEMELIHARAAN] Updating pemeliharaan ID: $id for user: $current_user_id");

        $input = getInputData();

        // Validation
        $required_fields = ['kebun_id', 'kegiatan', 'tanggal', 'jumlah', 'biaya'];
        $missing_fields = [];

        foreach ($required_fields as $field) {
            if (!isset($input[$field]) || $input[$field] === '' || $input[$field] === null) {
                $missing_fields[] = $field;
            }
        }

        if (!empty($missing_fields)) {
            sendResponse("error", "Field berikut wajib diisi: " . implode(', ', $missing_fields), null, 400, "MISSING_REQUIRED_FIELDS");
        }

        // Cek apakah pemeliharaan exists dan milik user yang login
        $cek = $conn->prepare("SELECT p.id FROM data_pemeliharaan p JOIN data_kebun l ON p.kebun_id = l.id WHERE p.id = ? AND l.user_id = ?");
        $cek->bind_param("ii", $id, $current_user_id);
        $cek->execute();
        $cek->store_result();

        if ($cek->num_rows == 0) {
            sendResponse("error", "Data pemeliharaan tidak ditemukan atau bukan milik Anda", null, 404, "PEMELIHARAAN_NOT_FOUND");
        }

        // Ambil data input
        $kebun_id = (int)$input['kebun_id'];
        $kegiatan = trim($input['kegiatan']);
        $tanggal_raw = trim($input['tanggal']);
        $jumlah = (int)$input['jumlah'];
        $satuan = isset($input['satuan']) ? trim($input['satuan']) : null;
        $biaya = (int)$input['biaya'];
        $catatan = isset($input['catatan']) ? trim($input['catatan']) : null;

        // Convert date format
        $tanggal = convertDateFormat($tanggal_raw);
        if (!$tanggal) {
            sendResponse("error", "Format tanggal tidak valid. Gunakan format DD-MM-YYYY atau YYYY-MM-DD", null, 400, "INVALID_DATE_FORMAT");
        }

        // Validasi kebun_id milik user yang login
        $cek_kebun = $conn->prepare("SELECT nama, lokasi FROM data_kebun WHERE id = ? AND user_id = ?");
        $cek_kebun->bind_param("ii", $kebun_id, $current_user_id);
        $cek_kebun->execute();
        $cek_result = $cek_kebun->get_result();

        if ($cek_result->num_rows == 0) {
            sendResponse("error", "Kebun tidak ditemukan atau bukan milik Anda", null, 404, "KEBUN_NOT_FOUND");
        }

        $kebun_data = $cek_result->fetch_assoc();

        // Additional validation
        if (strlen($kegiatan) < 3) {
            sendResponse("error", "Kegiatan minimal 3 karakter", null, 400, "INVALID_KEGIATAN");
        }

        if ($jumlah <= 0) {
            sendResponse("error", "Jumlah harus lebih dari 0", null, 400, "INVALID_JUMLAH");
        }

        if ($biaya < 0) {
            sendResponse("error", "Biaya tidak boleh negatif", null, 400, "INVALID_BIAYA");
        }

        // ✅ UPDATE: SQL query yang benar dengan security check
        $sql = "UPDATE data_pemeliharaan p
                JOIN data_kebun l ON p.kebun_id = l.id
                SET p.kebun_id = ?, 
                    p.kegiatan = ?, 
                    p.tanggal = ?, 
                    p.jumlah = ?, 
                    p.satuan = ?, 
                    p.biaya = ?, 
                    p.catatan = ? 
                WHERE p.id = ? 
                AND l.user_id = ?";

        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            error_log("❌ [PEMELIHARAAN] Update prepare failed: " . $conn->error);
            sendResponse("error", "Database error: " . $conn->error, null, 500, "DATABASE_ERROR");
        }

        $stmt->bind_param(
            "issisisii",
            $kebun_id,
            $kegiatan,
            $tanggal,
            $jumlah,
            $satuan,
            $biaya,
            $catatan,
            $id,
            $current_user_id
        );

        $result = $stmt->execute();

        if ($result && $stmt->affected_rows > 0) {
            error_log("✅ [PEMELIHARAAN] Successfully updated pemeliharaan ID: $id");

            $response_data = [
                "id" => $id,
                "kebun_id" => $kebun_id,
                "kegiatan" => $kegiatan,
                "tanggal" => $tanggal,
                "jumlah" => $jumlah,
                "satuan" => $satuan,
                "biaya" => $biaya,
                "catatan" => $catatan,
                "nama_kebun" => $kebun_data['nama'],
                "lokasi_kebun" => $kebun_data['lokasi'],
                "nama_user" => $user_payload['nama']
            ];

            sendResponse("success", "Data pemeliharaan berhasil diupdate", $response_data, 200);
        } else if ($stmt->affected_rows == 0) {
            error_log("⚠️ [PEMELIHARAAN] No changes made to pemeliharaan ID: $id");
            sendResponse("error", "Tidak ada perubahan data atau data tidak ditemukan", null, 400, "NO_CHANGES");
        } else {
            error_log("❌ [PEMELIHARAAN] Update failed: " . $stmt->error);
            sendResponse("error", "Gagal mengupdate data pemeliharaan: " . $stmt->error, null, 500, "UPDATE_FAILED");
        }
    } catch (Exception $e) {
        error_log("💥 [PEMELIHARAAN] PUT Exception: " . $e->getMessage());
        sendResponse("error", "Terjadi kesalahan sistem: " . $e->getMessage(), null, 500, "SYSTEM_ERROR");
    }
}


// DELETE - Delete pemeliharaan
if ($method == 'DELETE') {
    try {
        if (!$id) {
            sendResponse("error", "ID pemeliharaan diperlukan untuk delete", null, 400, "MISSING_ID");
        }

        error_log("🗑️ [PEMELIHARAAN] Deleting pemeliharaan ID: $id for user: $current_user_id");

        // Cek apakah pemeliharaan exists dan milik user yang login
        $cek = $conn->prepare("SELECT p.id FROM data_pemeliharaan p JOIN data_kebun l ON p.kebun_id = l.id WHERE p.id = ? AND l.user_id = ?");
        $cek->bind_param("ii", $id, $current_user_id);
        $cek->execute();
        $cek->store_result();

        if ($cek->num_rows == 0) {
            sendResponse("error", "Data pemeliharaan tidak ditemukan atau bukan milik Anda", null, 404, "PEMELIHARAAN_NOT_FOUND");
        }

        // Hapus pemeliharaan
        $sql = "DELETE p FROM data_pemeliharaan p JOIN data_kebun l ON p.kebun_id = l.id WHERE p.id = ? AND l.user_id = ?";
        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            error_log("❌ [PEMELIHARAAN] Delete prepare failed: " . $conn->error);
            sendResponse("error", "Database error", null, 500, "DATABASE_ERROR");
        }

        $stmt->bind_param("ii", $id, $current_user_id);
        $result = $stmt->execute();

        if ($result && $stmt->affected_rows > 0) {
            error_log("✅ [PEMELIHARAAN] Successfully deleted pemeliharaan ID: $id");
            sendResponse("success", "Data pemeliharaan berhasil dihapus");
        } else {
            error_log("❌ [PEMELIHARAAN] Delete failed or no rows affected");
            sendResponse("error", "Data pemeliharaan tidak ditemukan atau bukan milik Anda", null, 404, "PEMELIHARAAN_NOT_FOUND");
        }
    } catch (Exception $e) {
        error_log("💥 [PEMELIHARAAN] DELETE Exception: " . $e->getMessage());
        sendResponse("error", "Terjadi kesalahan sistem", null, 500, "SYSTEM_ERROR");
    }
}

// Method not allowed
error_log("❌ [PEMELIHARAAN] Method not allowed: $method");
sendResponse("error", "Method tidak didukung", null, 405, "METHOD_NOT_ALLOWED");
