<?php
date_default_timezone_set('Asia/Jakarta');
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

$endpoint = isset($_GET['endpoint']) ? $_GET['endpoint'] : '';

// Function to get input data
function getInputData()
{
    $input = file_get_contents('php://input');
    return json_decode($input, true) ?? [];
}

//REQUEST RESET PASSWORD
if ($endpoint == 'request-reset') {
    $input = getInputData();
    $email = isset($input['email']) ? trim($input['email']) : '';

    if (empty($email)) {
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Email wajib diisi",
            "error_code" => "MISSING_EMAIL"
        ]);
        exit;
    }

    // ✅ CEK RATE LIMITING - Batasi request OTP (misal: max 1 request per 2 menit)
    $stmt_check_rate = $conn->prepare("SELECT terakhir_request FROM reset_kata_sandi WHERE email = ? ORDER BY tanggal_dibuat DESC LIMIT 1");
    $stmt_check_rate->bind_param("s", $email);
    $stmt_check_rate->execute();
    $result_rate = $stmt_check_rate->get_result();

    if ($result_rate->num_rows > 0) {
        $rate_data = $result_rate->fetch_assoc();
        if ($rate_data['terakhir_request'] !== null) {
            $terakhir_request = new DateTime($rate_data['terakhir_request'], new DateTimeZone('Asia/Jakarta'));
            $now = new DateTime('now', new DateTimeZone('Asia/Jakarta'));
            $selisih_detik = $now->getTimestamp() - $terakhir_request->getTimestamp();

            // Batasi request minimal 2 menit (120 detik)
            $cooldown_detik = 120;
            if ($selisih_detik < $cooldown_detik) {
                $sisa_detik = $cooldown_detik - $selisih_detik;
                $sisa_menit = ceil($sisa_detik / 60);

                error_log("⏱️ [RATE-LIMIT] Too many requests from: $email (waited only $selisih_detik seconds)");

                http_response_code(429);
                echo json_encode([
                    "status" => "error",
                    "message" => "Mohon tunggu $sisa_menit menit sebelum request kode OTP baru",
                    "error_code" => "RATE_LIMIT_EXCEEDED",
                    "data" => [
                        "sisa_detik" => $sisa_detik,
                        "cooldown_detik" => $cooldown_detik
                    ]
                ]);
                exit;
            }
        }
    }

    //Generate OTP without leading zeros issues
    $otp_kode = str_pad((string)mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);

    // Set timezone consistently
    $timezone = new DateTimeZone('Asia/Jakarta');
    $now = new DateTime('now', $timezone);
    $berakhir = clone $now;
    $berakhir->add(new DateInterval('PT10M')); // +10 minutes

    $berakhir_pada = $berakhir->format('Y-m-d H:i:s');
    $tanggal_dibuat = $now->format('Y-m-d H:i:s');
    $terakhir_request = $tanggal_dibuat; // ✅ Set terakhir_request sama dengan tanggal_dibuat

    error_log("🔐 [REQUEST-RESET] Generated OTP: '$otp_kode' for: $email");
    error_log("⏰ [REQUEST-RESET] Created: $tanggal_dibuat");
    error_log("⏰ [REQUEST-RESET] Expires: $berakhir_pada");
    error_log("⏰ [REQUEST-RESET] Last request: $terakhir_request");

    // Cek apakah email terdaftar
    $stmt = $conn->prepare("SELECT id, nama FROM users WHERE email = ?");
    $stmt->bind_param("s", $email);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows == 0) {
        http_response_code(404);
        echo json_encode([
            "status" => "error",
            "message" => "Email tidak terdaftar",
            "error_code" => "EMAIL_NOT_FOUND"
        ]);
        exit;
    }

    $user = $result->fetch_assoc();
    $user_id = $user['id'];
    $nama = $user['nama'];

    //HAPUS OTP LAMA UNTUK EMAIL INI SEBELUM GENERATE YANG BARU
    $stmt_delete_old = $conn->prepare("DELETE FROM reset_kata_sandi WHERE email = ?");
    $stmt_delete_old->bind_param("s", $email);
    $stmt_delete_old->execute();
    error_log("🗑️ [RESET-PASSWORD] Deleted old OTPs for: $email");

    error_log("🔐 [RESET-PASSWORD] Generated OTP: $otp_kode for user: $email");

    // ✅ Simpan OTP ke database dengan percobaan = 0 dan terakhir_request
    $percobaan_gagal = 0;
    $stmt_insert = $conn->prepare("INSERT INTO reset_kata_sandi (user_id, email, otp_kode, berakhir_pada, tanggal_dibuat, terakhir_request, percobaan_gagal) VALUES (?, ?, ?, ?, ?, ?, ?)");
    $stmt_insert->bind_param("isssssi", $user_id, $email, $otp_kode, $berakhir_pada, $tanggal_dibuat, $terakhir_request, $percobaan_gagal);

    if (!$stmt_insert->execute()) {
        error_log("❌ [RESET-PASSWORD] Failed to insert OTP: " . $stmt_insert->error);
        http_response_code(500);
        echo json_encode([
            "status" => "error",
            "message" => "Gagal menyimpan OTP",
            "error_code" => "DATABASE_ERROR"
        ]);
        exit;
    }

    // Kirim OTP via Email menggunakan PHPMailer
    try {
        $mail = new PHPMailer(true);

        $mail->isSMTP();
        $mail->Host       = Config::get('SMTP_HOST');
        $mail->SMTPAuth   = true;
        $mail->Username   = Config::get('SMTP_USERNAME');
        $mail->Password   = Config::get('SMTP_PASSWORD');
        $mail->SMTPSecure = Config::get('SMTP_ENCRYPTION') === 'STARTTLS'
            ? PHPMailer::ENCRYPTION_STARTTLS
            : PHPMailer::ENCRYPTION_SMTPS;
        $mail->Port       = (int)Config::get('SMTP_PORT');

        $mail->SMTPOptions = [
            'ssl' => [
                'verify_peer' => false,
                'verify_peer_name' => false,
                'allow_self_signed' => true
            ]
        ];

        $mail->setFrom(Config::get('SMTP_USERNAME'), 'Tani App');
        $mail->addAddress($email, $nama);
        $mail->isHTML(true);
        $mail->Subject = 'Reset Password - Kode OTP';

        // Email Body
        $mail->Body = "
        <html>
        <head>
            <style>
                body { font-family: Arial, sans-serif; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background-color: #4CAF50; color: white; padding: 20px; text-align: center; }
                .content { padding: 20px; background-color: #f9f9f9; }
                .otp-code { font-size: 32px; font-weight: bold; color: #4CAF50; text-align: center; padding: 20px; background-color: #e8f5e9; border-radius: 5px; margin: 20px 0; }
                .footer { text-align: center; color: #666; font-size: 12px; padding: 20px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h2>Reset Password</h2>
                </div>
                <div class='content'>
                    <p>Halo <strong>$nama</strong>,</p>
                    <p>Anda telah meminta untuk mereset password akun Anda. Gunakan kode OTP berikut:</p>
                    <div class='otp-code'>$otp_kode</div>
                    <p><strong>Kode ini berlaku selama 10 menit.</strong></p>
                    <p><strong>Anda memiliki maksimal 5 kali percobaan untuk memasukkan kode.</strong></p>
                    <p>Jika Anda tidak meminta reset password, abaikan email ini.</p>
                </div>
                <div class='footer'>
                    <p>© 2025 Tani App. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>
        ";

        $mail->AltBody = "Halo $nama,\n\nKode OTP Anda: $otp_kode\n\nKode ini berlaku selama 10 menit.\nAnda memiliki maksimal 5 kali percobaan untuk memasukkan kode.";

        $mail->send();
        error_log("✅ [RESET-PASSWORD] Email sent successfully to: $email");

        echo json_encode([
            "status" => "success",
            "message" => "Kode OTP telah dikirim ke email Anda",
            "data" => [
                "email" => $email,
                "berakhir_pada" => $berakhir_pada,
                "max_percobaan" => 5,
                "cooldown_detik" => 120
            ]
        ]);
    } catch (Exception $e) {
        error_log("❌ [RESET-PASSWORD] Email failed: " . $mail->ErrorInfo);
        http_response_code(500);
        echo json_encode([
            "status" => "error",
            "message" => "Gagal mengirim email: " . $mail->ErrorInfo,
            "error_code" => "EMAIL_SEND_FAILED"
        ]);
    }

    exit;
}

//ENDPOINT: VERIFY OTP
if ($endpoint == 'verify-otp') {
    $input = getInputData();

    // ✅ Clean input properly
    $email = isset($input['email']) ? trim($input['email']) : '';
    $otp_kode = isset($input['otp_kode']) ? trim($input['otp_kode']) : '';

    // ✅ Remove any whitespace
    $otp_kode = preg_replace('/\s+/', '', $otp_kode);

    error_log("=== VERIFY OTP DEBUG START ===");
    error_log("📥 Raw input: " . json_encode($input));
    error_log("📧 Email: '$email'");
    error_log("🔐 OTP (cleaned): '$otp_kode' (length: " . strlen($otp_kode) . ")");

    // ✅ Validate before query
    if (empty($email) || empty($otp_kode)) {
        error_log("❌ Missing email or OTP");
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Email dan kode OTP wajib diisi",
            "error_code" => "MISSING_FIELDS"
        ]);
        exit;
    }

    // ✅ Validate OTP format (must be exactly 6 digits)
    if (!preg_match('/^\d{6}$/', $otp_kode)) {
        error_log("❌ Invalid OTP format: '$otp_kode'");
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Kode OTP harus 6 digit angka",
            "error_code" => "INVALID_OTP_FORMAT"
        ]);
        exit;
    }

    // ✅ Query untuk mendapatkan OTP data termasuk percobaan_gagal
    $stmt = $conn->prepare("SELECT id, user_id, email, otp_kode, berakhir_pada, tanggal_dibuat, terakhir_request, percobaan_gagal 
                            FROM reset_kata_sandi 
                            WHERE email = ? 
                            ORDER BY tanggal_dibuat DESC 
                            LIMIT 1");
    $stmt->bind_param("s", $email);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows == 0) {
        error_log("❌ [VERIFY-OTP] No OTP record found for email: $email");
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Tidak ada permintaan reset password untuk email ini",
            "error_code" => "NO_OTP_REQUEST"
        ]);
        exit;
    }

    $otp_data = $result->fetch_assoc();

    // ✅ Enhanced logging
    error_log("📊 Database OTP found:");
    error_log("   - Email: '{$otp_data['email']}'");
    error_log("   - OTP in DB: '{$otp_data['otp_kode']}'");
    error_log("   - Created: {$otp_data['tanggal_dibuat']}");
    error_log("   - Last request: {$otp_data['terakhir_request']}");
    error_log("   - Expires: {$otp_data['berakhir_pada']}");
    error_log("   - Failed attempts: {$otp_data['percobaan_gagal']}");

    // ✅ CEK APAKAH SUDAH MELEBIHI BATAS PERCOBAAN
    $max_percobaan = 5;
    if ($otp_data['percobaan_gagal'] >= $max_percobaan) {
        error_log("❌ [VERIFY-OTP] Max attempts exceeded for: $email");

        // Hapus OTP yang sudah melebihi batas
        $stmt_delete = $conn->prepare("DELETE FROM reset_kata_sandi WHERE id = ?");
        $stmt_delete->bind_param("i", $otp_data['id']);
        $stmt_delete->execute();

        http_response_code(403);
        echo json_encode([
            "status" => "error",
            "message" => "Anda telah melewati batas maksimal percobaan. Silakan request kode OTP baru.",
            "error_code" => "MAX_ATTEMPTS_EXCEEDED"
        ]);
        exit;
    }

    // ✅ Check expiry with proper timezone handling
    $now = new DateTime('now', new DateTimeZone('Asia/Jakarta'));
    $berakhir_pada = new DateTime($otp_data['berakhir_pada'], new DateTimeZone('Asia/Jakarta'));

    error_log("⏰ Time check:");
    error_log("   - Now:        " . $now->format('Y-m-d H:i:s'));
    error_log("   - Expires at: " . $berakhir_pada->format('Y-m-d H:i:s'));
    error_log("   - Is expired: " . ($now > $berakhir_pada ? 'YES ❌' : 'NO ✅'));

    if ($now > $berakhir_pada) {
        error_log("❌ [VERIFY-OTP] OTP expired");

        // ✅ Clean expired OTP
        $stmt_delete = $conn->prepare("DELETE FROM reset_kata_sandi WHERE id = ?");
        $stmt_delete->bind_param("i", $otp_data['id']);
        $stmt_delete->execute();

        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Kode OTP sudah kadaluarsa",
            "error_code" => "OTP_EXPIRED"
        ]);
        exit;
    }

    // ✅ CEK APAKAH OTP COCOK
    if ($otp_data['otp_kode'] !== $otp_kode) {
        error_log("❌ [VERIFY-OTP] OTP mismatch");

        // ✅ INCREMENT percobaan_gagal
        $percobaan_baru = $otp_data['percobaan_gagal'] + 1;
        $sisa_percobaan = $max_percobaan - $percobaan_baru;

        $stmt_update = $conn->prepare("UPDATE reset_kata_sandi SET percobaan_gagal = ? WHERE id = ?");
        $stmt_update->bind_param("ii", $percobaan_baru, $otp_data['id']);
        $stmt_update->execute();

        error_log("   - Failed attempts updated: $percobaan_baru / $max_percobaan");
        error_log("   - Remaining attempts: $sisa_percobaan");

        // Jika sudah mencapai batas, hapus OTP
        if ($percobaan_baru >= $max_percobaan) {
            $stmt_delete = $conn->prepare("DELETE FROM reset_kata_sandi WHERE id = ?");
            $stmt_delete->bind_param("i", $otp_data['id']);
            $stmt_delete->execute();

            http_response_code(403);
            echo json_encode([
                "status" => "error",
                "message" => "Kode OTP salah. Anda telah melewati batas maksimal percobaan. Silakan request kode OTP baru.",
                "error_code" => "MAX_ATTEMPTS_EXCEEDED"
            ]);
            exit;
        }

        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Kode OTP salah. Sisa percobaan: $sisa_percobaan kali",
            "error_code" => "INVALID_OTP",
            "data" => [
                "sisa_percobaan" => $sisa_percobaan,
                "max_percobaan" => $max_percobaan
            ]
        ]);
        exit;
    }

    // ✅ OTP BENAR
    error_log("✅ [VERIFY-OTP] OTP verified successfully");
    error_log("=== VERIFY OTP DEBUG END ===");

    echo json_encode([
        "status" => "success",
        "message" => "Kode OTP valid. Silakan reset password Anda.",
        "data" => [
            "email" => $email,
            "berakhir_pada" => $otp_data['berakhir_pada']
        ]
    ]);
    exit;
}

// ✅ ENDPOINT: RESET PASSWORD (SETELAH OTP VERIFIED)
if ($endpoint == 'reset-password') {
    $input = getInputData();
    $email = isset($input['email']) ? trim($input['email']) : '';
    $otp_kode = isset($input['otp_kode']) ? trim($input['otp_kode']) : '';
    $new_password = isset($input['new_password']) ? $input['new_password'] : '';
    $confirm_password = isset($input['confirm_password']) ? $input['confirm_password'] : '';

    if (empty($email) || empty($otp_kode) || empty($new_password) || empty($confirm_password)) {
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Semua field wajib diisi",
            "error_code" => "MISSING_FIELDS"
        ]);
        exit;
    }

    if ($new_password !== $confirm_password) {
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Password dan konfirmasi password tidak cocok",
            "error_code" => "PASSWORD_MISMATCH"
        ]);
        exit;
    }

    error_log("🔐 [RESET-PASSWORD] Verifying OTP for email: $email");

    // ✅ Verifikasi OTP sekali lagi dengan cek percobaan_gagal
    $stmt = $conn->prepare("SELECT id, user_id, email, otp_kode, berakhir_pada, terakhir_request, percobaan_gagal 
                            FROM reset_kata_sandi 
                            WHERE email = ? AND otp_kode = ? 
                            ORDER BY tanggal_dibuat DESC 
                            LIMIT 1");
    $stmt->bind_param("ss", $email, $otp_kode);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows == 0) {
        error_log("❌ [RESET-PASSWORD] Invalid OTP for email: $email");
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Kode OTP tidak valid atau sudah kadaluarsa",
            "error_code" => "INVALID_OTP"
        ]);
        exit;
    }

    $otp_data = $result->fetch_assoc();

    // ✅ Cek apakah sudah melebihi batas percobaan
    if ($otp_data['percobaan_gagal'] >= 5) {
        error_log("❌ [RESET-PASSWORD] Max attempts exceeded");

        // Hapus OTP
        $stmt_delete = $conn->prepare("DELETE FROM reset_kata_sandi WHERE id = ?");
        $stmt_delete->bind_param("i", $otp_data['id']);
        $stmt_delete->execute();

        http_response_code(403);
        echo json_encode([
            "status" => "error",
            "message" => "Kode OTP sudah tidak valid. Silakan request kode baru.",
            "error_code" => "MAX_ATTEMPTS_EXCEEDED"
        ]);
        exit;
    }

    // Cek expired
    $now = new DateTime('now', new DateTimeZone('Asia/Jakarta'));
    $berakhir_pada = new DateTime($otp_data['berakhir_pada'], new DateTimeZone('Asia/Jakarta'));

    if ($now > $berakhir_pada) {
        error_log("❌ [RESET-PASSWORD] OTP expired for email: $email");

        // Hapus OTP expired
        $stmt_delete = $conn->prepare("DELETE FROM reset_kata_sandi WHERE id = ?");
        $stmt_delete->bind_param("i", $otp_data['id']);
        $stmt_delete->execute();

        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Kode OTP sudah kadaluarsa",
            "error_code" => "OTP_EXPIRED"
        ]);
        exit;
    }

    // ✅ Hash password baru
    $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);

    // ✅ Update kolom password_hash
    $stmt_update = $conn->prepare("UPDATE users SET password_hash = ? WHERE email = ?");
    $stmt_update->bind_param("ss", $hashed_password, $email);

    if (!$stmt_update->execute()) {
        error_log("❌ [RESET-PASSWORD] Failed to update password for: $email");
        http_response_code(500);
        echo json_encode([
            "status" => "error",
            "message" => "Gagal mereset password",
            "error_code" => "DATABASE_ERROR"
        ]);
        exit;
    }

    // ✅ HAPUS OTP SETELAH PASSWORD BERHASIL DIRESET
    $stmt_delete = $conn->prepare("DELETE FROM reset_kata_sandi WHERE email = ?");
    $stmt_delete->bind_param("s", $email);
    $stmt_delete->execute();

    error_log("✅ [RESET-PASSWORD] Password reset successful for: $email");

    echo json_encode([
        "status" => "success",
        "message" => "Password berhasil direset. Silakan login dengan password baru Anda.",
        "data" => [
            "email" => $email
        ]
    ]);
    exit;
}

// Endpoint not found
http_response_code(404);
echo json_encode([
    "status" => "error",
    "message" => "Endpoint tidak ditemukan",
    "error_code" => "ENDPOINT_NOT_FOUND"
]);
